/*
 *  linux/drivers/video/pm3fb.c -- 3DLabs Permedia3 frame buffer device
 *  
 *  Copyright (C) 2001 Romain Dolbeau <dolbeau@irisa.fr>
 *  Based on code written by:
 *           Sven Luther, <luther@dpt-info.u-strasbg.fr>
 *           Alan Hourihane, <alanh@fairlite.demon.co.uk>
 *  Based on linux/drivers/video/skeletonfb.c:
 *	Copyright (C) 1997 Geert Uytterhoeven
 *  Based on linux/driver/video/pm2fb.c:
 *      Copyright (C) 1998-1999 Ilario Nardinocchi (nardinoc@CS.UniBO.IT)
 *      Copyright (C) 1999 Jakub Jelinek (jakub@redhat.com)
 *
 *  This file is subject to the terms and conditions of the GNU General Public
 *  License. See the file COPYING in the main directory of this archive for
 *  more details.
 *
 *  $Header: /home/pm3fb/pm3fb/pm3fb.c,v 1.12 2001/02/27 12:58:36 dolbeau Exp $
 *
 *  CHANGELOG:
 *  Tue Feb 27 14:01:36 CET 2001, v 1.0.5: fixes (1.0.4 was broken for 2.2), cleaning up
 *  Mon Feb 26 23:17:36 CET 2001, v 1.0.4: preliminary 2.4.x support, dropped (useless on pm3) partial product, more OF fix
 *  Mon Feb 26 20:59:05 CET 2001, v 1.0.3: No more shadow register (and wasted memory), endianess fix, use OF-preset resolution by default
 *  Wed Feb 21 22:09:30 CET 2001, v 1.0.2: Code cleaning for future multiboard support, better OF support, bugs fix
 *  Wed Feb 21 19:58:56 CET 2001, v 1.0.1: OpenFirmware support, fixed memory detection, better debug support, code cleaning
 *  Wed Feb 21 14:47:06 CET 2001, v 1.0.0: First working version
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/mm.h>
#include <linux/tty.h>
#include <linux/malloc.h>
#include <linux/vmalloc.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <asm/uaccess.h>
#include <linux/fb.h>
#include <linux/init.h>
#include <linux/version.h>

#include <video/fbcon.h>
#include <video/fbcon-mfb.h>
#include <video/fbcon-cfb2.h>
#include <video/fbcon-cfb4.h>
#include <video/fbcon-cfb8.h>
#include <video/fbcon-cfb16.h>
#include <video/fbcon-cfb24.h>
#include <video/fbcon-cfb32.h>

#include <asm/io.h>
#include <linux/pci.h>
#ifdef CONFIG_FB_OF
#include <asm/prom.h>
#endif

#include "pm3fb.h"

/* what kernel is this ? */
#if ((LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)) && (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)))
#define KERNEL_2_4
#endif

#if ((LINUX_VERSION_CODE >= KERNEL_VERSION(2,2,0)) && (LINUX_VERSION_CODE < KERNEL_VERSION(2,3,0))) 
#define KERNEL_2_2
#ifdef CONFIG_FB_OF
#define SUPPORT_FB_OF
#endif
#endif

#if (!defined(KERNEL_2_2)) && (!defined(KERNEL_2_4))
#error "Only kernel 2.2.x and kernel 2.4.y might work"
#endif

/* not sure if/why it's needed. doesn't work without on my PowerMac... */
#ifdef __BIG_ENDIAN
#define MUST_BYTESWAP
#endif

/* ************************************* */
/* ***** The various "global" data ***** */
/* ************************************* */

/* those will need a rework for multiple board support */
/* Driver name */
static const char permedia3_name[16]="Permedia3";

/* the fb_par struct, mandatory */
struct pm3fb_par {
  u32 pixclock;		/* pixclock in KHz */
  
  u32 width;		/* width of virtual screen */
  u32 height;		/* height of virtual screen */
  
  u32 hsstart;		/* horiz. sync start */
  u32 hsend;		/* horiz. sync end */
  u32 hbend;		/* horiz. blank end (also gate end) */
  u32 htotal;		/* total width (w/ sync & blank) */
  
  u32 vsstart;		/* vert. sync start */
  u32 vsend;		/* vert. sync end */
  u32 vbend;		/* vert. blank end */
  u32 vtotal;		/* total height (w/ sync & blank) */
  
  u32 stride;		/* screen stride */
  u32 base;		/* screen base (xoffset+yoffset) */
  u32 depth;		/* screen depth (8, 16, 24 or 32) */
  u32 video;		/* video control (hsync,vsync) */
 };

/* the fb_info struct, mandatory */
struct pm3fb_info
{
  struct fb_info_gen gen;
  unsigned long board_num;
  int current_par_valid;
  unsigned long use_current;
  struct pm3fb_par *current_par;
  struct pci_dev *dev;
#ifdef SUPPORT_FB_OF
  struct device_node *dn;
#endif /* SUPPORT_FB_OF */
  unsigned char* fb_base;	/* framebuffer memory base */
  u32 fb_size;		/* framebuffer memory size */
  unsigned char* p_fb;	/* physical address of frame buffer */
  unsigned char* v_fb;	/* virtual address of frame buffer */
  unsigned char *pIOBase;/* physical address of registers region, must be rg_base or rg_base+PM2_REGS_SIZE depending on the host endianness */
  unsigned char *vIOBase; /* address of registers afer ioremap() */
  unsigned char BppShift;
  struct {
	u8 transp;
	u8 red;
	u8 green;
	u8 blue;
  } palette[256];
	union {
#ifdef FBCON_HAS_CFB16
		u16 cmap16[16];
#endif
#ifdef FBCON_HAS_CFB32
		u32 cmap32[16];
#endif
	} cmap;
};

/* regular resolution database*/
static struct
{
  char name[16];
  struct pm3fb_par user_mode;
} mode_base[] __initdata =
{
  {"800x600-75", {49500,800,600,16,96,256,1056,1,4,25,625,800,0,8,
		  PM3VideoControl_ENABLE|
		  PM3VideoControl_HSYNC_ACTIVE_HIGH|
		  PM3VideoControl_VSYNC_ACTIVE_HIGH|
		  PM3VideoControl_PIXELSIZE_8BIT}},
  {"1024x768-74", {77825,1024,768,16,176,276,1300,4,9,41,809,1024,0,8,
		   PM3VideoControl_ENABLE|
		   PM3VideoControl_HSYNC_FORCE_HIGH|
		   PM3VideoControl_VSYNC_FORCE_HIGH|
		   PM3VideoControl_PIXELSIZE_8BIT}},
  {"\0", }
};

/* more mandatory stuff (see skeletonfb.c + framebuffer driver HOWTO */
static struct pm3fb_info fb_info[PM3_MAX_BOARD];
static unsigned long detected_board = 0;
static struct pm3fb_par current_par[PM3_MAX_BOARD];
static struct display disp[PM3_MAX_BOARD];

/* really, inside display */
/* static struct fb_var_screeninfo default_var; */

static int currcon = 0;
static int inverse = 0;


/* ******************************************** */
/* ***** A bunch of register-access macro ***** */
/* ******************************************** */
#define PM3_WAIT(n) \
do{ \
	while(PM3_READ_REG(PM3InFIFOSpace)<(n)); \
} while(0)

#define PM3_SLOW_WRITE_REG(r,v)	\
do{                             \
    DASSERT((l_fb_info->vIOBase != (unsigned char*)(-1)), "l_fb_info->vIOBase mapped in slow write\n"); \
	mb();                   \
	PM3_WAIT(1);            \
	mb();                   \
    PM3_WRITE_REG(r,v);     \
} while(0)

#define PM3_SET_INDEX(index) \
do{ \
	PM3_SLOW_WRITE_REG(PM3RD_IndexHigh,(((index)>>8)&0xff)); \
	PM3_SLOW_WRITE_REG(PM3RD_IndexLow,((index)&0xff)); \
} while(0)

#ifdef MUST_BYTESWAP
#define PM3_READ_REG(r) __swab32(readl((l_fb_info->vIOBase + r)))
#define PM3_WRITE_REG(r, v) writel(__swab32(v), (l_fb_info->vIOBase + r))
#else /* MUST_BYTESWAP */
#define PM3_WRITE_REG(r, v) writel(v, (l_fb_info->vIOBase + r))
#define PM3_READ_REG(r) readl((l_fb_info->vIOBase + r))
#endif /* MUST_BYTESWAP */

#define PM3_WRITE_DAC_REG(r, v) \
do { \
     DASSERT((l_fb_info->vIOBase != (unsigned char*)(-1)), "l_fb_info->vIOBase mapped in write dac reg\n"); \
     PM3_SET_INDEX(r); \
     mb(); \
     PM3_WRITE_REG(PM3RD_IndexedData, v); \
} while (0)

#define PM3_READ_DAC_REG(r) pm3_read_dac_reg(l_fb_info, r)

static unsigned long pm3_read_dac_reg(struct pm3fb_info *l_fb_info, unsigned long r)
{
  DASSERT((l_fb_info->vIOBase != (unsigned char*)(-1)), "l_fb_info->vIOBase mapped in write dac reg\n"); \
  PM3_SET_INDEX(r);
  mb();
  return(PM3_READ_REG(PM3RD_IndexedData));
}

/* ************************************** */
/* ***** permedia-specific function ***** */
/* ************************************** */

/* Calculating various clock parameter */
static unsigned long pm3fb_CalculateClock(struct pm3fb_info *l_fb_info,
					  unsigned long reqclock, /* In kHz units */
					  unsigned long refclock, /* In kHz units */
					  unsigned char *prescale,/* ClkPreScale */
					  unsigned char *feedback,/* ClkFeedBackScale */
					  unsigned char *postscale/* ClkPostScale */)
{
    int			f, pre, post;
    unsigned long	freq;
    long		freqerr = 1000;
    unsigned long  	actualclock = 0;

    DTRACE;

    for (f=1;f<256;f++)
      {
	for (pre=1;pre<256;pre++)
	  {
	    for (post=0;post<5;post++)
	      { 
		freq = ((2* refclock * f) / (pre * (1 << post)));
		if ((reqclock > freq - freqerr)&&(reqclock < freq + freqerr))
		  {
		    freqerr = (reqclock > freq) ? 
		      reqclock - freq : freq - reqclock;
		    *feedback = f;
		    *prescale = pre;
		    *postscale = post;
		    actualclock = freq;
		  }
	      }
	  }
      }
    
    return(actualclock);
}

#define PICOS2KHZ(a) (1000000000UL/(a))
#define KHZ2PICOS(a) (1000000000UL/(a))

static int pm3fb_Shiftbpp(struct pm3fb_info *l_fb_info, unsigned long depth, int v)
{
  DTRACE;

  switch (depth)
    {
    case 8:
      l_fb_info->BppShift = 4;
      return(v >> 4);
    case 16:
      l_fb_info->BppShift = 3;
      return(v >> 3);
    case 32:
      l_fb_info->BppShift = 2;
      return(v >> 2);
    }
  DPRINTK(1, "Unsupported depth\n");
  return(0);
}

static int pm3fb_Unshiftbpp(struct pm3fb_info *l_fb_info, unsigned long depth, int v)
{
  DTRACE;

  switch (depth)
    {
    case 8:
      l_fb_info->BppShift = 4;
      return(v << 4);
    case 16:
      l_fb_info->BppShift = 3;
      return(v << 3);
    case 32:
      l_fb_info->BppShift = 2;
      return(v << 2);
    }
  DPRINTK(1, "Unsupported depth\n");
  return(0);
}

static void pm3_mapIO(struct pm3fb_info *l_fb_info)
{
  DTRACE;

  l_fb_info->vIOBase = ioremap((unsigned long)l_fb_info->pIOBase, 0x20000);
  l_fb_info->v_fb = ioremap((unsigned long)l_fb_info->p_fb, l_fb_info->fb_size);
  DPRINTK(2, "IO mapping : IOBase %lx / %lx, fb %lx / %lx\n",
	  (unsigned long)l_fb_info->pIOBase,
	  (unsigned long)l_fb_info->vIOBase,
	  (unsigned long)l_fb_info->p_fb,
	  (unsigned long)l_fb_info->v_fb);
}

static void pm3_unmapIO(struct pm3fb_info *l_fb_info)
{
  DTRACE;

  iounmap(l_fb_info->vIOBase);
  iounmap(l_fb_info->v_fb);
  l_fb_info->vIOBase = (unsigned char *)-1;
  l_fb_info->v_fb = (unsigned char *)-1;
}

#if defined(PM3FB_MASTER_DEBUG) && (PM3FB_MASTER_DEBUG >= 2)
#define PM3_SHOW_CUR_MODE pm3_show_cur_mode(l_fb_info);
#define PM3_SHOW_CUR_TIMING pm3_show_cur_timing(l_fb_info);
static void pm3_show_cur_mode(struct pm3fb_info *l_fb_info)
{
  DPRINTK(2, "PM3Aperture0: 0x%08x\n", PM3_READ_REG(PM3Aperture0));
  DPRINTK(2, "PM3Aperture1: 0x%08x\n", PM3_READ_REG(PM3Aperture1));
  DPRINTK(2, "PM3ByAperture1Mode: 0x%08x\n", PM3_READ_REG(PM3ByAperture1Mode));
  DPRINTK(2, "PM3ByAperture2Mode: 0x%08x\n", PM3_READ_REG(PM3ByAperture2Mode));
  DPRINTK(2, "PM3ChipConfig: 0x%08x\n", PM3_READ_REG(PM3ChipConfig));
  DPRINTK(2, "PM3FIFODis: 0x%08x\n", PM3_READ_REG(PM3FIFODis));
  DPRINTK(2, "PM3HTotal: 0x%08x\n", PM3_READ_REG(PM3HTotal));
  DPRINTK(2, "PM3HbEnd: 0x%08x\n", PM3_READ_REG(PM3HbEnd));
  DPRINTK(2, "PM3HgEnd: 0x%08x\n", PM3_READ_REG(PM3HgEnd));
  DPRINTK(2, "PM3HsEnd: 0x%08x\n", PM3_READ_REG(PM3HsEnd));
  DPRINTK(2, "PM3HsStart: 0x%08x\n", PM3_READ_REG(PM3HsStart));
  DPRINTK(2, "PM3MemBypassWriteMask: 0x%08x\n", PM3_READ_REG(PM3MemBypassWriteMask));
  DPRINTK(2, "PM3RD_IndexControl: 0x%08x\n", PM3_READ_REG(PM3RD_IndexControl));
  DPRINTK(2, "PM3ScreenBase: 0x%08x\n", PM3_READ_REG(PM3ScreenBase));
  DPRINTK(2, "PM3ScreenStride: 0x%08x\n", PM3_READ_REG(PM3ScreenStride));
  DPRINTK(2, "PM3VClkCtl: 0x%08x\n", PM3_READ_REG(PM3VClkCtl));
  DPRINTK(2, "PM3VTotal: 0x%08x\n", PM3_READ_REG(PM3VTotal));
  DPRINTK(2, "PM3VbEnd: 0x%08x\n", PM3_READ_REG(PM3VbEnd));
  DPRINTK(2, "PM3VideoControl: 0x%08x\n", PM3_READ_REG(PM3VideoControl));
  DPRINTK(2, "PM3VsEnd: 0x%08x\n", PM3_READ_REG(PM3VsEnd));
  DPRINTK(2, "PM3VsStart: 0x%08x\n", PM3_READ_REG(PM3VsStart));

  DPRINTK(2, "PM3RD_ColorFormat: %ld\n", PM3_READ_DAC_REG(PM3RD_ColorFormat));
  DPRINTK(2, "PM3RD_DACControl: %ld\n", PM3_READ_DAC_REG(PM3RD_DACControl));
  DPRINTK(2, "PM3RD_DClk0FeedbackScale: %ld\n", PM3_READ_DAC_REG(PM3RD_DClk0FeedbackScale));
  DPRINTK(2, "PM3RD_DClk0PostScale: %ld\n", PM3_READ_DAC_REG(PM3RD_DClk0PostScale));
  DPRINTK(2, "PM3RD_DClk0PreScale: %ld\n", PM3_READ_DAC_REG(PM3RD_DClk0PreScale));
  DPRINTK(2, "[not set] PM3RD_IndexControl: %ld\n", PM3_READ_DAC_REG(PM3RD_IndexControl));
  DPRINTK(2, "PM3RD_MiscControl: %ld\n", PM3_READ_DAC_REG(PM3RD_MiscControl));
  DPRINTK(2, "PM3RD_PixelSize: %ld\n", PM3_READ_DAC_REG(PM3RD_PixelSize));
  DPRINTK(2, "PM3RD_SyncControl: %ld\n", PM3_READ_DAC_REG(PM3RD_SyncControl));
}

static void  pm3_show_cur_timing(struct pm3fb_info *l_fb_info)
{
  DPRINTK(2, "PM3LocalMemCaps: 0x%08x\n", PM3_READ_REG(PM3LocalMemCaps));
  DPRINTK(2, "PM3LocalMemTimings: 0x%08x\n", PM3_READ_REG(PM3LocalMemTimings));
  DPRINTK(2, "PM3LocalMemControl: 0x%08x\n", PM3_READ_REG(PM3LocalMemControl));
  DPRINTK(2, "PM3LocalMemRefresh: 0x%08x\n", PM3_READ_REG(PM3LocalMemRefresh));
  DPRINTK(2, "PM3LocalMemPowerDown: 0x%08x\n", PM3_READ_REG(PM3LocalMemPowerDown));
}
#else
#define PM3_SHOW_CUR_MODE /* pm3_show_mode(); */
#define PM3_SHOW_CUR_TIMING /* pm3_show_cur_timing(); */
#endif

/* write the mode to registers */
static void pm3_write_mode(struct pm3fb_info *l_fb_info)
{
  DTRACE;

  PM3_SLOW_WRITE_REG(PM3MemBypassWriteMask,  	0xffffffff);
  PM3_SLOW_WRITE_REG(PM3Aperture0,		 	0x00000000);
  PM3_SLOW_WRITE_REG(PM3Aperture1,		 	0x00000000);
  PM3_SLOW_WRITE_REG(PM3FIFODis,			0x00000007);

  PM3_SLOW_WRITE_REG(PM3HTotal,
		     pm3fb_Shiftbpp(l_fb_info, l_fb_info->current_par->depth,
				    l_fb_info->current_par->htotal - 1));
  PM3_SLOW_WRITE_REG(PM3HsEnd,
		     pm3fb_Shiftbpp(l_fb_info, l_fb_info->current_par->depth,
				    l_fb_info->current_par->hsend));
  PM3_SLOW_WRITE_REG(PM3HsStart,
		     pm3fb_Shiftbpp(l_fb_info, l_fb_info->current_par->depth,
				    l_fb_info->current_par->hsstart));
  PM3_SLOW_WRITE_REG(PM3HbEnd,
		     pm3fb_Shiftbpp(l_fb_info, l_fb_info->current_par->depth,
				    l_fb_info->current_par->hbend));
  PM3_SLOW_WRITE_REG(PM3HgEnd,
		     pm3fb_Shiftbpp(l_fb_info, l_fb_info->current_par->depth,
				    l_fb_info->current_par->hbend));
  PM3_SLOW_WRITE_REG(PM3ScreenStride,
		     pm3fb_Shiftbpp(l_fb_info, l_fb_info->current_par->depth,
				    l_fb_info->current_par->stride));
  PM3_SLOW_WRITE_REG(PM3VTotal, l_fb_info->current_par->vtotal - 1);
  PM3_SLOW_WRITE_REG(PM3VsEnd, l_fb_info->current_par->vsend - 1);
  PM3_SLOW_WRITE_REG(PM3VsStart, l_fb_info->current_par->vsstart - 1);
  PM3_SLOW_WRITE_REG(PM3VbEnd, l_fb_info->current_par->vbend);

  switch (l_fb_info->current_par->depth)
    {
    case 8:
      PM3_SLOW_WRITE_REG(PM3ByAperture1Mode, PM3ByApertureMode_PIXELSIZE_8BIT);
      PM3_SLOW_WRITE_REG(PM3ByAperture2Mode, PM3ByApertureMode_PIXELSIZE_8BIT);
      break;

    case 16:
#ifndef __BIG_ENDIAN
      PM3_SLOW_WRITE_REG(PM3ByAperture1Mode, PM3ByApertureMode_PIXELSIZE_16BIT);
      PM3_SLOW_WRITE_REG(PM3ByAperture2Mode, PM3ByApertureMode_PIXELSIZE_16BIT);
#else
      PM3_SLOW_WRITE_REG(PM3ByAperture1Mode, PM3ByApertureMode_PIXELSIZE_16BIT |
					PM3ByApertureMode_BYTESWAP_BADC);
      PM3_SLOW_WRITE_REG(PM3ByAperture2Mode, PM3ByApertureMode_PIXELSIZE_16BIT |
					PM3ByApertureMode_BYTESWAP_BADC);
#endif
      break;

    case 32:
#ifndef __BIG_ENDIAN
      PM3_SLOW_WRITE_REG(PM3ByAperture1Mode, PM3ByApertureMode_PIXELSIZE_32BIT);
      PM3_SLOW_WRITE_REG(PM3ByAperture2Mode, PM3ByApertureMode_PIXELSIZE_32BIT);
#else
      PM3_SLOW_WRITE_REG(PM3ByAperture1Mode, PM3ByApertureMode_PIXELSIZE_32BIT |
		    PM3ByApertureMode_BYTESWAP_DCBA);
      PM3_SLOW_WRITE_REG(PM3ByAperture2Mode, PM3ByApertureMode_PIXELSIZE_32BIT |
		    PM3ByApertureMode_BYTESWAP_DCBA);
#endif
      break;

	default:
	  DPRINTK(1, "Unsupported depth\n");
	  break;
    }

  PM3_SLOW_WRITE_REG(PM3VideoControl, l_fb_info->current_par->video);
  PM3_SLOW_WRITE_REG(PM3VClkCtl, (PM3_READ_REG(PM3VClkCtl) & 0xFFFFFFFC));
  PM3_SLOW_WRITE_REG(PM3ScreenBase, l_fb_info->current_par->base);
  PM3_SLOW_WRITE_REG(PM3ChipConfig, (PM3_READ_REG(PM3ChipConfig) & 0xFFFFFFFD));
  
  {
    unsigned char m;		/* ClkPreScale */
    unsigned char n; 		/* ClkFeedBackScale */
    unsigned char p;		/* ClkPostScale */
    unsigned long usedclock = pm3fb_CalculateClock(l_fb_info, l_fb_info->current_par->pixclock, PM3_REF_CLOCK, &m, &n, &p); /* usedclock unused */

    DPRINTK(2, "Pixclock: %ld, Pre: %d, Feedback: %d, Post: %d\n",
	    l_fb_info->current_par->pixclock,
	    (int)m, (int)n, (int)p);

    PM3_WRITE_DAC_REG(PM3RD_DClk0PreScale, m);
    PM3_WRITE_DAC_REG(PM3RD_DClk0FeedbackScale, n);
    PM3_WRITE_DAC_REG(PM3RD_DClk0PostScale, p);
  }
  /*
  PM3_WRITE_DAC_REG(PM3RD_IndexControl, 0x00);
  */
  /*
  PM3_SLOW_WRITE_REG(PM3RD_IndexControl, 0x00);
  */
  {
    char tempsync = 0x00;
    
    if ((l_fb_info->current_par->video & PM3VideoControl_HSYNC_MASK) == PM3VideoControl_HSYNC_ACTIVE_HIGH)
      tempsync |= PM3RD_SyncControl_HSYNC_ACTIVE_HIGH;
    if ((l_fb_info->current_par->video & PM3VideoControl_VSYNC_MASK) == PM3VideoControl_VSYNC_ACTIVE_HIGH)
      tempsync |= PM3RD_SyncControl_VSYNC_ACTIVE_HIGH;
    
    PM3_WRITE_DAC_REG(PM3RD_SyncControl, tempsync);
    DPRINTK(2, "PM3RD_SyncControl: %d\n", tempsync);
  }
  PM3_WRITE_DAC_REG(PM3RD_DACControl, 0x00);
  
  switch (l_fb_info->current_par->depth)
    {
    case 8:
      PM3_WRITE_DAC_REG(PM3RD_PixelSize, PM3RD_PixelSize_8_BIT_PIXELS);
      PM3_WRITE_DAC_REG(PM3RD_ColorFormat,
						PM3RD_ColorFormat_CI8_COLOR|
						PM3RD_ColorFormat_COLOR_ORDER_BLUE_LOW);
      break;
    case 16:
      PM3_WRITE_DAC_REG(PM3RD_PixelSize, PM3RD_PixelSize_16_BIT_PIXELS);
      PM3_WRITE_DAC_REG(PM3RD_ColorFormat,
						PM3RD_ColorFormat_565_FRONT_COLOR|
						PM3RD_ColorFormat_COLOR_ORDER_BLUE_LOW|
						PM3RD_ColorFormat_LINEAR_COLOR_EXT_ENABLE);
      break;
    case 32:
      PM3_WRITE_DAC_REG(PM3RD_PixelSize, PM3RD_PixelSize_32_BIT_PIXELS);
      PM3_WRITE_DAC_REG(PM3RD_ColorFormat,
						PM3RD_ColorFormat_8888_COLOR|
						PM3RD_ColorFormat_COLOR_ORDER_BLUE_LOW);
      break;
    }

  PM3_WRITE_DAC_REG(PM3RD_MiscControl, PM3RD_MiscControl_HIGHCOLOR_RES_ENABLE);

  PM3_SHOW_CUR_MODE
}

static void pm3_read_mode(struct pm3fb_info *l_fb_info, struct pm3fb_par *curpar)
{
  unsigned long pixsize1, pixsize2, clockused;
  unsigned long pre, feedback, post;

  DTRACE;

  clockused = PM3_READ_REG(PM3VClkCtl);

  switch (clockused)
    {
    case 3:
      pre = PM3_READ_DAC_REG(PM3RD_DClk3PreScale);
      feedback = PM3_READ_DAC_REG(PM3RD_DClk3FeedbackScale);
      post = PM3_READ_DAC_REG(PM3RD_DClk3PostScale);
      
      DPRINTK(2, "DClk3 parameter: Pre: %ld, Feedback: %ld, Post: %ld ; giving pixclock: %ld\n",
	      pre, feedback, post,
	      PM3_SCALE_TO_CLOCK(pre, feedback, post) );
      break;
    case 2:
      pre = PM3_READ_DAC_REG(PM3RD_DClk2PreScale);
      feedback = PM3_READ_DAC_REG(PM3RD_DClk2FeedbackScale);
      post = PM3_READ_DAC_REG(PM3RD_DClk2PostScale);
      
      DPRINTK(2, "DClk2 parameter: Pre: %ld, Feedback: %ld, Post: %ld ; giving pixclock: %ld\n",
	      pre, feedback, post,
	      PM3_SCALE_TO_CLOCK(pre, feedback, post) );
      break;
    case 1:
      pre = PM3_READ_DAC_REG(PM3RD_DClk1PreScale);
      feedback = PM3_READ_DAC_REG(PM3RD_DClk1FeedbackScale);
      post = PM3_READ_DAC_REG(PM3RD_DClk1PostScale);
      
      DPRINTK(2, "DClk1 parameter: Pre: %ld, Feedback: %ld, Post: %ld ; giving pixclock: %ld\n",
	      pre, feedback, post,
	      PM3_SCALE_TO_CLOCK(pre, feedback, post) );
      break;
    case 0:
      pre = PM3_READ_DAC_REG(PM3RD_DClk0PreScale);
      feedback = PM3_READ_DAC_REG(PM3RD_DClk0FeedbackScale);
      post = PM3_READ_DAC_REG(PM3RD_DClk0PostScale);
      
      DPRINTK(2, "DClk0 parameter: Pre: %ld, Feedback: %ld, Post: %ld ; giving pixclock: %ld\n",
	      pre, feedback, post,
	      PM3_SCALE_TO_CLOCK(pre, feedback, post) );
      break;
    default:
      pre = feedback = post = 0;
      DPRINTK(1, "Unknowk D clock used : %ld\n", clockused);
      break;
    }

  curpar->pixclock = PM3_SCALE_TO_CLOCK(pre, feedback, post);
      
  pixsize1 = PM3ByApertureMode_PIXELSIZE_MASK & (PM3_READ_REG(PM3ByAperture1Mode));
  pixsize2 = PM3ByApertureMode_PIXELSIZE_MASK & (PM3_READ_REG(PM3ByAperture2Mode));

  DASSERT((pixsize1 == pixsize2), "pixsize the same in both aperture\n");

  if (pixsize1 & PM3ByApertureMode_PIXELSIZE_32BIT)
    curpar->depth = 32;
  else
    if (pixsize1 & PM3ByApertureMode_PIXELSIZE_16BIT)
      curpar->depth = 16;
    else
      curpar->depth = 8;

  /* not sure if I need to add one on the next ; it give better result with */
  curpar->htotal = pm3fb_Unshiftbpp(l_fb_info, curpar->depth, 1 + PM3_READ_REG(PM3HTotal));
  curpar->hsend = pm3fb_Unshiftbpp(l_fb_info, curpar->depth, PM3_READ_REG(PM3HsEnd));
  curpar->hsstart = pm3fb_Unshiftbpp(l_fb_info, curpar->depth, PM3_READ_REG(PM3HsStart));
  curpar->hbend = pm3fb_Unshiftbpp(l_fb_info, curpar->depth, PM3_READ_REG(PM3HbEnd));

  curpar->stride = pm3fb_Unshiftbpp(l_fb_info, curpar->depth, PM3_READ_REG(PM3ScreenStride));

  curpar->vtotal = 1 + PM3_READ_REG(PM3VTotal);
  curpar->vsend = 1 + PM3_READ_REG(PM3VsEnd);
  curpar->vsstart = 1 + PM3_READ_REG(PM3VsStart);
  curpar->vbend = PM3_READ_REG(PM3VbEnd);

  curpar->video = PM3_READ_REG(PM3VideoControl);

  curpar->base = 0; /* PM3_READ_REG(PM3ScreenBase); */
  curpar->width = curpar->htotal - curpar->hbend;
  curpar->height = curpar->vtotal - curpar->vbend;

  DPRINTK(2, "Found : %d * %d, %d Khz, stride is %08x\n",
	  curpar->width, curpar->height, curpar->pixclock, curpar->stride);
}

static unsigned long pm3_size_memory(struct pm3fb_info *l_fb_info)
{
  unsigned long memsize, tempBypass, i, temp1, temp2;
  
  DTRACE;

  l_fb_info->fb_size = 64*1024*1024; /* pm3 aperture alwayx 64 MB */
  pm3_mapIO(l_fb_info); /* temporary map IO */
  
  DASSERT((l_fb_info->vIOBase != NULL), "IO successfully mapped before mem detect\n");
  DASSERT((l_fb_info->v_fb != NULL), "FBsuccessfully mapped before mem detect\n");

  tempBypass = PM3_READ_REG(PM3MemBypassWriteMask);

  DPRINTK(2, "PM3MemBypassWriteMask was: 0x%08lx\n", tempBypass);

  PM3_SLOW_WRITE_REG(PM3MemBypassWriteMask, 0xFFFFFFFF);

  /* pm3 split up memory, replicates, and do a lot pof nasty stuff IMHO ;-) */

  for(i = 0; i < 32; i++)
	{
#ifdef MUST_BYTESWAP
	  writel(__swab32(i * 0x00345678), (l_fb_info->v_fb + (i * 1048576)));
#else
	  writel(i * 0x00345678, (l_fb_info->v_fb + (i * 1048576)));
#endif
	  mb();
#ifdef MUST_BYTESWAP
	  temp1 = __swab32(readl((l_fb_info->v_fb + (i * 1048576))));
#else
	  temp1 = readl((l_fb_info->v_fb + (i * 1048576)));
#endif
	  /* Let's check for wrapover, write will fail at 16MB boundary */
	  if (temp1 == (i * 0x00345678)) 
		memsize = i;
	  else 
		break;
	}

  DPRINTK(2, "First detect pass already got %ld MB\n", memsize + 1);

  /* ok we've done the first 32 mb... */
  
  if (memsize == i)
	{
	  for(i=0;i<32;i++)
		{
		  /* Clear first 32MB ; 0 is 0, no need to byteswap */
		  writel(0x0000000, (l_fb_info->v_fb + (i * 1048576)));
		  mb();
		}

	  for(i=32;i<64;i++)
		{
#ifdef MUST_BYTESWAP
		  writel(__swab32(i * 0x00345678), (l_fb_info->v_fb + (i * 1048576)));
#else
		  writel(i * 0x00345678, (l_fb_info->v_fb + (i * 1048576)));
#endif
		  mb();
#ifdef MUST_BYTESWAP
		  temp1 = __swab32(readl((l_fb_info->v_fb + (i * 1048576))));
		  temp2 = __swab32(readl((l_fb_info->v_fb + ((i - 32) * 1048576))));
#else
		  temp1 = readl((l_fb_info->v_fb + (i * 1048576)));
		  temp2 = readl((l_fb_info->v_fb + ((i - 32) * 1048576)));
#endif
		  if ( (temp1 == (i*0x00345678)) && (temp2 == 0) ) /* different value, different RAM... */
			memsize = i;
		  else 
			break;
		}
	}

  DPRINTK(2, "Second detect pass got %ld MB\n", memsize + 1);

  PM3_SLOW_WRITE_REG(PM3MemBypassWriteMask, tempBypass);

  pm3_unmapIO(l_fb_info);
  memsize = 1048576 * (memsize + 1);
  
  DPRINTK(2, "Returning 0x%08lx bytes\n", memsize);

  l_fb_info->fb_size = memsize;

  return memsize;
}

static void pm3fb_mode_setup(char *mode, unsigned long board_num)
{
  struct pm3fb_info *l_fb_info = &(fb_info[board_num]);
  struct pm3fb_par *l_fb_par = &(current_par[board_num]);
  unsigned long i = 0;

  if (!strncmp(mode, "current", 7))
	{
	  l_fb_info->use_current = 1; /* default w/ OpenFirmware */
	}
  else
	{
	  while (mode_base[i].user_mode.width)
		{
		  if (!(strcmp(mode, mode_base[i].name)))
			{
			  memcpy(l_fb_par, &(mode_base[i].user_mode), sizeof(struct pm3fb_par));
			  l_fb_info->current_par_valid = 1;
			  break;
			}
		}
	  DASSERT(l_fb_info->current_par_valid, "Valid mode on command line\n");
	}
}

/* ********************************************** */
/* ***** framebuffer API standard functions ***** */
/* ********************************************** */

static int pm3fb_open(struct fb_info* info, int user)
{
  struct pm3fb_info *l_fb_info = (struct pm3fb_info*) info;

  DTRACE;

  DPRINTK(3, "user is %d\n", user);

  MOD_INC_USE_COUNT;
 
  return(0);
}

static int pm3fb_release(struct fb_info* info, int user)
{
  struct pm3fb_info *l_fb_info = (struct pm3fb_info*) info;

  DTRACE;

  MOD_DEC_USE_COUNT;

  return(0);
}

static int pm3fb_encode_fix(struct fb_fix_screeninfo *fix,
			    const void *par,
			    struct fb_info_gen *info)
{
  struct pm3fb_info* l_fb_info=(struct pm3fb_info* )info;
  struct pm3fb_par* p=(struct pm3fb_par* )par;

  DTRACE;
  
  strcpy(fix->id, permedia3_name);
  fix->smem_start = l_fb_info->p_fb;
  fix->smem_len = l_fb_info->fb_size;
  fix->mmio_start = l_fb_info->pIOBase;
  fix->mmio_len = PM3_REGS_SIZE;
  fix->accel = 0 /* FB_ACCEL_3DLABS_PERMEDIA3 */;
  fix->type = FB_TYPE_PACKED_PIXELS;
  fix->visual= (p->depth == 8) ? FB_VISUAL_PSEUDOCOLOR:FB_VISUAL_TRUECOLOR;
  if (l_fb_info->current_par_valid)
	fix->line_length = l_fb_info->current_par->width*(l_fb_info->current_par->depth/8);
  else
	fix->line_length = 0;
  fix->xpanstep = 64/p->depth;
  fix->ypanstep = 1;
  fix->ywrapstep = 0;
  return 0;
}



static int pm3fb_decode_var(const struct fb_var_screeninfo* var,
			    void *par,
			    struct fb_info_gen* info)
{
  struct pm3fb_info *l_fb_info = (struct pm3fb_info*)info;
  struct pm3fb_par *p = (struct pm3fb_par*)par;
  struct pm3fb_par temp_p;
  u32 xres;
  
  DTRACE;

  DASSERT((var != NULL), "fb_var_screeninfo* not NULL");
  DASSERT((p != NULL), "pm3fb_par* not NULL");
  DASSERT((l_fb_info != NULL), "pm3fb_info* not NULL");

  memset(&temp_p, 0, sizeof(struct pm3fb_par));
  temp_p.width = (var->xres_virtual + 7) & ~7;
  temp_p.height = var->yres_virtual;
  temp_p.depth = (var->bits_per_pixel + 7) & ~7;
  temp_p.depth = (temp_p.depth > 32) ? 32 : temp_p.depth;

  DPRINTK(2, "xres: %d, yres: %d, vxres: %d, vyres: %d ; xoffset:%d, yoffset: %d\n",
	  var->xres, var->yres, var->xres_virtual, var->yres_virtual, var->xoffset, var->yoffset);

  xres = (var->xres + 31) & ~31;
  if (temp_p.width<xres+var->xoffset)
    temp_p.width=xres+var->xoffset;
  if (temp_p.height<var->yres+var->yoffset)
    temp_p.height=var->yres+var->yoffset;

  DPRINTK(3, "First var filling done\n");
    
  if (temp_p.width>2048) {
    DPRINTK(1, "virtual width not supported: %u\n", temp_p.width);
    return -EINVAL;
  }
  if (var->yres<200) {
    DPRINTK(1, "height not supported: %u\n",
	    (u32 )var->yres);
    return -EINVAL;
  }
  if (temp_p.height<200 || temp_p.height>2048) {
    DPRINTK(1, "virtual height not supported: %u\n", temp_p.height);
    return -EINVAL;
  }
  if (temp_p.depth>32) {
    DPRINTK(1, "depth not supported: %u\n", temp_p.depth);
    return -EINVAL;
  }
  if (temp_p.width*temp_p.height*temp_p.depth/8>l_fb_info->fb_size) {
    DPRINTK(1, "no memory for screen (%ux%ux%u)\n",
	    temp_p.width, temp_p.height, temp_p.depth);
    return -EINVAL;
  }

  if ((!var->pixclock) ||
      (!var->right_margin) ||
      (!var->hsync_len) ||
      (!var->left_margin) ||
      (!var->lower_margin) ||
      (!var->vsync_len) ||
      (!var->upper_margin)
      )
    {
      unsigned long i = 0, done = 0;
      printk("pm3fb: refusing to use a likely wrong timing\n");
      
      while ((mode_base[i].user_mode.width) && !done)
	{
	  if ((mode_base[i].user_mode.width == temp_p.width) &&
	      (mode_base[i].user_mode.height == temp_p.height))
	    {
	      printk("pm3fb: using close match %s\n", mode_base[i].name);
	      temp_p = mode_base[i].user_mode;
	      done = 1;
	    }
	  i++;
	}
      if (!done)
	return -EINVAL;
    }
  else
    {
      temp_p.pixclock=PICOS2KHZ(var->pixclock);
      if (temp_p.pixclock>PM3_MAX_PIXCLOCK)
	{
	  DPRINTK(1, "pixclock too high (%uKHz)\n", temp_p.pixclock);
	  return -EINVAL;
	}
      
      DPRINTK(3, "Clock done\n");
      
      temp_p.hsstart = var->right_margin;
      temp_p.hsend = var->right_margin + var->hsync_len;
      temp_p.hbend = var->right_margin + var->hsync_len + var->left_margin;
      temp_p.htotal = xres + temp_p.hbend;
      
      temp_p.vsstart = var->lower_margin;
      temp_p.vsend = var->lower_margin + var->vsync_len;
      temp_p.vbend = var->lower_margin + var->vsync_len + var->upper_margin;
      temp_p.vtotal = var->yres + temp_p.vbend;
      
      temp_p.stride = temp_p.width;

      DPRINTK(2, "Using %d * %d, %d Khz, stride is %08x\n",
	      temp_p.width, temp_p.height, temp_p.pixclock, temp_p.stride);
      
      temp_p.base = (var->yoffset * xres) + var->xoffset;
      
      temp_p.video = 0;
      
      if (var->sync & FB_SYNC_HOR_HIGH_ACT)
	temp_p.video |= PM3VideoControl_HSYNC_ACTIVE_HIGH;
      else
	temp_p.video |= PM3VideoControl_HSYNC_ACTIVE_LOW;
      
      if (var->sync & FB_SYNC_VERT_HIGH_ACT)
	temp_p.video |= PM3VideoControl_VSYNC_ACTIVE_HIGH;
      else
	temp_p.video |= PM3VideoControl_VSYNC_ACTIVE_LOW;
      
      if ((var->vmode & FB_VMODE_MASK) == FB_VMODE_INTERLACED)
	{
	  DPRINTK(1, "interlaced not supported\n\n");
	  return -EINVAL;
	}
      
      if ((var->vmode & FB_VMODE_MASK) == FB_VMODE_DOUBLE)
	temp_p.video |= PM3VideoControl_LINE_DOUBLE_ON;
      else
	temp_p.video |= PM3VideoControl_LINE_DOUBLE_OFF;
      
      if (var->activate == FB_ACTIVATE_NOW)
	temp_p.video |= PM3VideoControl_ENABLE;
      else
	{
	  temp_p.video |= PM3VideoControl_DISABLE;
	  DPRINTK(2, "PM3Video disabled\n");
	}
      
      switch (temp_p.depth)
	{
	case 8:
	  temp_p.video |= PM3VideoControl_PIXELSIZE_8BIT;
	  break;
	case 16:
	  temp_p.video |= PM3VideoControl_PIXELSIZE_16BIT;
	  break;
	case 32:
	  temp_p.video |= PM3VideoControl_PIXELSIZE_32BIT;
	  break;
	default:
	  DPRINTK(1, "Unsupported depth\n");
	  break;
	}
    }
  (*p) = temp_p; 
  
  DPRINTK(3, "All done\n");
  
  return 0;
}

static int pm3fb_encode_var(struct fb_var_screeninfo *var,
			    const void *par,
			    struct fb_info_gen *info)
{
  struct pm3fb_par* p= (struct pm3fb_par*) par;
  struct pm3fb_info *l_fb_info = (struct pm3fb_info*) info;

  u32 base;
  
  DTRACE; 

  DASSERT((var != NULL), "fb_var_screeninfo* not NULL");
  DASSERT((p != NULL), "pm3fb_par* not NULL");
  DASSERT((info != NULL), "fb_info_gen* not NULL");
 
  memset(var, 0, sizeof(struct fb_var_screeninfo));
  /*
    if (p->flags & PM2FF_ACCEL)
    var->accel_flags |= FB_ACCELF_TEXT;
  */
  var->xres_virtual = p->width;
  var->yres_virtual = p->height;
  var->xres = p->htotal - p->hbend;
  var->yres = p->vtotal - p->vbend;

  DPRINTK(2, "xres = %d, yres : %d\n", var->xres, var->yres);

  var->right_margin = p->hsstart;
  var->hsync_len = p->hsend - p->hsstart;
  var->left_margin = p->hbend - p->hsend;
  var->lower_margin = p->vsstart;
  var->vsync_len = p->vsend - p->vsstart;
  var->upper_margin = p->vbend - p->vsend;
  var->bits_per_pixel = p->depth;
  
  switch (p->depth) {
  case 8:
    var->red.length = var->green.length = var->blue.length = 8;
    break;

  case 16:
    var->red.offset = 11;
    var->red.length = 5;
    var->green.offset = 5;
    var->green.length = 6;
    var->blue.length = 5;
    break;

  case 32:
    var->transp.offset = 24;
    var->red.offset = 16;
    var->green.offset = 8;
    var->red.length = var->green.length = var->blue.length = var->transp.length = 8;
    break;

  default:
    DPRINTK(1, "Unsupported depth\n");
    break;
  }

  base = p->base;

  var->xoffset = base % var->xres;
  var->yoffset = base / var->xres;

  var->height = var->width = -1;

  var->pixclock = KHZ2PICOS(p->pixclock);

  DPRINTK(3, "Clock done\n");

  if ((p->video & PM3VideoControl_HSYNC_MASK) == PM3VideoControl_HSYNC_ACTIVE_HIGH)
    var->sync |= FB_SYNC_HOR_HIGH_ACT;
  if ((p->video & PM3VideoControl_VSYNC_MASK) == PM3VideoControl_VSYNC_ACTIVE_HIGH)
    var->sync |= FB_SYNC_VERT_HIGH_ACT;
  if (p->video & PM3VideoControl_LINE_DOUBLE_ON)
    var->vmode = FB_VMODE_DOUBLE;

  DPRINTK(3, "all done\n");

  return 0;
}

static void pm3fb_get_par(void *par, struct fb_info_gen *info)
{
  struct pm3fb_info *l_fb_info = (struct pm3fb_info*) info;

  DTRACE;

  if (!l_fb_info->current_par_valid) {
	if (l_fb_info->use_current)
	  pm3_read_mode(l_fb_info, l_fb_info->current_par);
	else
	  memcpy(l_fb_info->current_par, &(mode_base[0].user_mode), sizeof(struct pm3fb_par));
	l_fb_info->current_par_valid = 1;
  }
  *((struct pm3fb_par* )par) = *(l_fb_info->current_par);
}

static void pm3fb_set_par(const void *par, struct fb_info_gen *info)
{
  struct pm3fb_info *l_fb_info = (struct pm3fb_info*) info;

  DTRACE;
  
  *(l_fb_info->current_par)  =  *((struct pm3fb_par*)par);
  l_fb_info->current_par_valid = 1;

  pm3_write_mode(l_fb_info);
}

static void pm3fb_set_color(struct pm3fb_info* l_fb_info, unsigned char regno,
			unsigned char r, unsigned char g, unsigned char b)
{
  DTRACE;

  DPRINTK(3, "Writing color #%d, rgb %3d:%3d:%3d\n", (int)regno, (int)r, (int)g, (int)b);

  PM3_SLOW_WRITE_REG(PM3RD_PaletteWriteAddress, regno);
  PM3_SLOW_WRITE_REG(PM3RD_PaletteData, r);
  PM3_SLOW_WRITE_REG(PM3RD_PaletteData, g);
  PM3_SLOW_WRITE_REG(PM3RD_PaletteData, b);
}

static int pm3fb_getcolreg(unsigned regno, unsigned *red, unsigned *green,
			   unsigned *blue, unsigned *transp,
			   struct fb_info *info)
{
  struct pm3fb_info* l_fb_info = (struct pm3fb_info* )info;
  
  DTRACE;

  if (regno<256) {
    *red = l_fb_info->palette[regno].red<<8|l_fb_info->palette[regno].red;
    *green = l_fb_info->palette[regno].green<<8|l_fb_info->palette[regno].green;
    *blue = l_fb_info->palette[regno].blue<<8|l_fb_info->palette[regno].blue;
    *transp = l_fb_info->palette[regno].transp<<8|l_fb_info->palette[regno].transp;
  }
  return regno>255;
}

static int pm3fb_setcolreg(unsigned regno, unsigned red, unsigned green,
			   unsigned blue, unsigned transp,
			   struct fb_info *info)
{
  struct pm3fb_info* l_fb_info = (struct pm3fb_info* )info;
  
  DTRACE;
  
  if (regno<16) {
    switch (l_fb_info->current_par->depth) {
#ifdef FBCON_HAS_CFB8
    case 8:
      break;
#endif
#ifdef FBCON_HAS_CFB16
    case 16:
      l_fb_info->cmap.cmap16[regno]=
	((u32 )red & 0xf800) |
	(((u32 )green & 0xfc00)>>5) |
	(((u32 )blue & 0xf800)>>11);
      break;
#endif
#ifdef FBCON_HAS_CFB32
    case 32:
      l_fb_info->cmap.cmap32[regno]=
	(((u32 )transp & 0xff00) << 16) |
	(((u32 )red & 0xff00) << 8) |
	(((u32 )green & 0xff00)) |
	(((u32 )blue & 0xff00) >> 8);
      break;
#endif
    default:
      DPRINTK(1, "bad depth %u\n", l_fb_info->current_par->depth);
      break;
    }
  }
  if (regno<256) {
    l_fb_info->palette[regno].red=red >> 8;
    l_fb_info->palette[regno].green=green >> 8;
    l_fb_info->palette[regno].blue=blue >> 8;
    l_fb_info->palette[regno].transp=transp >> 8;
    if (l_fb_info->current_par->depth==8)
      pm3fb_set_color(l_fb_info, regno, red>>8, green>>8, blue>>8);
  }
  return regno>255;
}

static int pm3fb_blank(int blank_mode, struct fb_info_gen *info)
{
  struct pm3fb_info *l_fb_info = (struct pm3fb_info*) info;

  DTRACE;

  /* TO BE DONE */
#if 0
  struct pm2fb_info* i=(struct pm2fb_info* )info;
  u32 video;
  
  if (!l_fb_info->current_par_valid)
    return 1;
  video=current_par.video;
  if (blank_mode>0) {
    switch (blank_mode-1) {
    case VESA_NO_BLANKING: /* FIXME */
      video=video&~(PM2F_VIDEO_ENABLE);
      break;
    case VESA_HSYNC_SUSPEND:
      video=video&~(PM2F_HSYNC_MASK|
		    PM2F_BLANK_LOW);
      break;
    case VESA_VSYNC_SUSPEND:
      video=video&~(PM2F_VSYNC_MASK|
		    PM2F_BLANK_LOW);
      break;
    case VESA_POWERDOWN:
      video=video&~(PM2F_VSYNC_MASK|
		    PM2F_HSYNC_MASK|
		    PM2F_BLANK_LOW);
      break;
    }
  }
  WAIT_FIFO(i, 1);
  pm2_WR(i, PM2R_VIDEO_CONTROL, video);
  return 0;
#endif /* 0 */
  return(1);
}

static void pm3fb_set_disp(const void *par, struct display *disp,
			   struct fb_info_gen *info)
{
  struct pm3fb_info *l_fb_info = (struct pm3fb_info *)info;
  struct pm3fb_par *p = (struct pm3fb_par *)par;
  u32 flags;

  DTRACE;
  
  save_flags(flags); cli();
  disp->screen_base=l_fb_info->v_fb;
  switch (p->depth)
    {
#ifdef FBCON_HAS_CFB8
    case 8:
      disp->dispsw=&fbcon_cfb8;
      break;
#endif
#ifdef FBCON_HAS_CFB16
    case 16:
      disp->dispsw=&fbcon_cfb16;
      disp->dispsw_data=l_fb_info->cmap.cmap16;
      break;
#endif
#ifdef FBCON_HAS_CFB32
    case 32:
      disp->dispsw=&fbcon_cfb32;
      disp->dispsw_data=l_fb_info->cmap.cmap32;
      break;
#endif
    default:
      disp->dispsw=&fbcon_dummy;
      DPRINTK(1, "Invalid depth, using fbcon_dummy\n");
      break;
    }
  restore_flags(flags);
}

/* */
static void pm3fb_detect(void /* struct pm3fb_info *l_fb_info */)
{
  struct pci_dev *dev = NULL;
  struct pm3fb_info *l_fb_info = &(fb_info[detected_board]);
  
  /* should be reworked (multiple boards / chips-perp-board...) */
  
  DTRACE;
  
  dev = pci_find_device(PCI_VENDOR_ID_3DLABS, PCI_DEVICE_ID_3DLABS_PERMEDIA3, dev);
  if (!dev)
    return;
  l_fb_info->dev = dev;
#ifdef SUPPORT_FB_OF
  l_fb_info->dn = NULL;
#endif

#ifdef KERNEL_2_2
  DPRINTK(1, "found @%lx Vendor %lx Device %lx ; base @ : %lx - %lx - %lx - %lx - %lx - %lx, rom @ %lx, irq %ld\n",
	  (unsigned long)dev,
	  (unsigned long)dev->vendor,
	  (unsigned long)dev->device,
	  (unsigned long)dev->base_address[0],
	  (unsigned long)dev->base_address[1], 
	  (unsigned long)dev->base_address[2],
	  (unsigned long)dev->base_address[3],
	  (unsigned long)dev->base_address[4],
	  (unsigned long)dev->base_address[5],
	  (unsigned long)dev->rom_address,
	  (unsigned long)dev->irq);

  l_fb_info->pIOBase = (unsigned char*)(l_fb_info->dev->base_address[0] & PCI_BASE_ADDRESS_MEM_MASK);
#ifdef __BIG_ENDIAN
  l_fb_info->pIOBase += PM3_REGS_SIZE;
#endif
  l_fb_info->vIOBase = (unsigned char*)-1;
  l_fb_info->p_fb = (unsigned char*)(l_fb_info->dev->base_address[1] & PCI_BASE_ADDRESS_MEM_MASK);
  l_fb_info->v_fb = (unsigned char*)-1;
  l_fb_info->fb_size = pm3_size_memory(l_fb_info);
#endif /* KERNEL_2_2 */
#ifdef KERNEL_2_4
  DPRINTK(1, "found @%lx Vendor %lx Device %lx ; base @ : %lx - %lx - %lx - %lx - %lx - %lx, rom @ %lx, irq %ld\n",
	  (unsigned long)dev,
	  (unsigned long)dev->vendor,
	  (unsigned long)dev->device,
	  (unsigned long)dev->resource[0].start,
	  (unsigned long)dev->resource[1].start, 
	  (unsigned long)dev->resource[2].start,
	  (unsigned long)dev->resource[3].start,
	  (unsigned long)dev->resource[4].start,
	  (unsigned long)dev->resource[5].start,
	  (unsigned long)dev->resource[6].start,
	  (unsigned long)dev->irq);

  l_fb_info->pIOBase = (unsigned char*)(l_fb_info->dev->resource[0].start & PCI_BASE_ADDRESS_MEM_MASK);
#ifdef __BIG_ENDIAN
  l_fb_info->pIOBase += PM3_REGS_SIZE;
#endif
  l_fb_info->vIOBase = (unsigned char*)-1;
  l_fb_info->p_fb = (unsigned char*)(l_fb_info->dev->resource[1].start & PCI_BASE_ADDRESS_MEM_MASK);
  l_fb_info->v_fb = (unsigned char*)-1;
  l_fb_info->fb_size = pm3_size_memory(l_fb_info);
#endif /* KERNEL_2_4 */

  pm3_mapIO(l_fb_info);

  PM3_SHOW_CUR_MODE

  PM3_SHOW_CUR_TIMING
	
  /*
  pm3fb_get_par(&par, &l_fb_info->gen.info);
  pm3fb_encode_var(&default_var, &par, &l_fb_info->gen.info);
  */
}

#ifdef KERNEL_2_2
struct fbgen_hwswitch pm3fb_switch = {
    pm3fb_detect, pm3fb_encode_fix, pm3fb_decode_var, pm3fb_encode_var,
    pm3fb_get_par, pm3fb_set_par, pm3fb_getcolreg, pm3fb_setcolreg,
    fbgen_pan_display, pm3fb_blank, pm3fb_set_disp
};

static struct fb_ops pm3fb_ops = {
    pm3fb_open, pm3fb_release,
    fbgen_get_fix, fbgen_get_var, fbgen_set_var,
    fbgen_get_cmap, fbgen_set_cmap, fbgen_pan_display, fbgen_ioctl, NULL, NULL
};
#endif /* KERNEL_2_2 */
#ifdef KERNEL_2_4
struct fbgen_hwswitch pm3fb_switch = {
    pm3fb_detect, pm3fb_encode_fix, pm3fb_decode_var, pm3fb_encode_var,
    pm3fb_get_par, pm3fb_set_par, pm3fb_getcolreg, pm3fb_setcolreg,
    fbgen_pan_display, pm3fb_blank, pm3fb_set_disp
};

static struct fb_ops pm3fb_ops = {
    THIS_MODULE,
    pm3fb_open, pm3fb_release,
    fbgen_get_fix, fbgen_get_var, fbgen_set_var,
    fbgen_get_cmap, fbgen_set_cmap, fbgen_pan_display, NULL, NULL, NULL
};
#endif /* KERNEL_2_4 */


/* common initialisation */
static void pm3fb_common_init(struct pm3fb_info *l_fb_info)
{
  strcpy(l_fb_info->gen.info.modename, permedia3_name);
  disp[l_fb_info->board_num].scrollmode=SCROLL_YNOMOVE;
  l_fb_info->gen.parsize=sizeof(struct pm3fb_par);
  l_fb_info->gen.info.changevar = NULL;
  l_fb_info->gen.info.node = -1;
  l_fb_info->gen.info.fbops = &pm3fb_ops;
  l_fb_info->gen.info.disp = &(disp[l_fb_info->board_num]);
  l_fb_info->gen.info.switch_con = &fbgen_switch;
  l_fb_info->gen.info.updatevar = &fbgen_update_var; /* */
  l_fb_info->gen.info.blank = &fbgen_blank; /* */
  l_fb_info->gen.info.flags = FBINFO_FLAG_DEFAULT;

  DPRINTK(3, "Var filling done\n");
  
  (void)fbgen_get_var(&(disp[l_fb_info->board_num]).var, -1, &l_fb_info->gen.info);
  
  DPRINTK(3, "First fbgen done\n");
  
  (void)fbgen_do_set_var(&(disp[l_fb_info->board_num]).var, 1, &l_fb_info->gen);
  
  DPRINTK(3, "Second fbgen done\n");

  fbgen_set_disp(-1, &l_fb_info->gen);

  DPRINTK(3, "Third fbgen done\n");

  fbgen_install_cmap(0, &l_fb_info->gen);

  DPRINTK(3, "Fourth fbgen done\n");

  if (register_framebuffer(&l_fb_info->gen.info) < 0)
	{
	  DPRINTK(1, "Couldn't register framebuffer\n");
	  return;
	}

  DPRINTK(3, "Register fb done\n");
  
  pm3_write_mode(l_fb_info);

  detected_board++;

  printk("fb%d: %s, using %uK of video memory.\n",
	 GET_FB_IDX(l_fb_info->gen.info.node),
	 permedia3_name,
	 (u32)(l_fb_info->fb_size>>10));
}

/* ****************************************** */
/* ***** standard FB API init functions ***** */
/* ****************************************** */

#ifdef KERNEL_2_4
int __init pm3fb_setup(char *options)
#endif
#ifdef KERNEL_2_2
__initfunc(void pm3fb_setup(char *options, int *ints))
#endif
{
  char *next;

  DTRACE;

  DPRINTK(2, "Options : %s\n", options);

  while (options) /*only board 0 for now */
	{            
	  if ((next=strchr(options, ',')))
		*(next++)='\0';
	  if (!strncmp(options, "mode:", 5))
		pm3fb_mode_setup(options+5, 0 /* board_num */);
	  options = next;
	}
#ifdef KERNEL_2_4
  return(0);
#endif
}

#ifdef KERNEL_2_4
int __init pm3fb_init(void)
#endif
#ifdef KERNEL_2_2
__initfunc(void pm3fb_init(void))
#endif
{
  struct pm3fb_info *l_fb_info = &(fb_info[detected_board]);

  DTRACE;

  memset(l_fb_info, 0, sizeof(struct pm3fb_info));
  
  l_fb_info->gen.fbhw = &pm3fb_switch;
  l_fb_info->gen.fbhw->detect();

  if (!l_fb_info->dev)
	{
	  DPRINTK(1, "No PCI Permedia3 board detected\n");
	  return;
	}

  l_fb_info->board_num = detected_board;

#ifdef SUPPORT_FB_OF
  {
    struct device_node *dp =
      find_pci_device_OFnode(l_fb_info->dev->bus->number, l_fb_info->dev->devfn);
    
    if (dp)
      {
		if (!strncmp(dp->name, "formacGA12",10)) /* we'll handle it in pm3fb_of_init */
		  {
			pm3_unmapIO(l_fb_info);
			return;
		  }
      }
  }
#endif

  l_fb_info->current_par = &(current_par[l_fb_info->board_num]);

  pm3fb_common_init(l_fb_info);

#ifdef KERNEL_2_4
  return(0);
#endif
}

#ifdef SUPPORT_FB_OF
__initfunc(void pm3fb_of_init(struct device_node *dp))
{
  struct pm3fb_info *l_fb_info = &(fb_info[detected_board]);

  DTRACE;

  DPRINTK(2, "OpenFirmware board : %s\n", dp->full_name);

  {
    long i;
    for (i = 0; i < dp->n_addrs; i++)
      {
	DPRINTK(2, "MemRange : 0x%08x - 0x%x\n", dp->addrs[i].address, dp->addrs[i].size);
      }
  }

  l_fb_info->board_num = detected_board;

  memset(l_fb_info, 0, sizeof(struct pm3fb_info));
  
  l_fb_info->gen.fbhw = &pm3fb_switch;
  l_fb_info->dev = NULL;
  l_fb_info->dn = dp;

  l_fb_info->current_par = &(current_par[l_fb_info->board_num]);

  l_fb_info->pIOBase = (unsigned char*)dp->addrs[3].address;
#ifdef __BIG_ENDIAN
  l_fb_info->pIOBase += PM3_REGS_SIZE;
#endif
  l_fb_info->vIOBase = (unsigned char*)-1;
  l_fb_info->p_fb = (unsigned char*)dp->addrs[1].address;
  l_fb_info->v_fb = (unsigned char*)-1;
  
  l_fb_info->fb_size = pm3_size_memory(l_fb_info); /* (unsigned long)dp->addrs[1].size; */ /* OF is a liar ! it claims 256 Mb */

  DPRINTK(2, "OpenFirmware board : IOBase 0x%08lx, p_fb 0x%08lx, fb_size %d KB\n",
	  (unsigned long)l_fb_info->pIOBase, (unsigned long)l_fb_info->p_fb, l_fb_info->fb_size >> 10);

  pm3_mapIO(l_fb_info);
  
  PM3_SHOW_CUR_MODE
    
  PM3_SHOW_CUR_TIMING

  l_fb_info->use_current = 1; /* will use current mode by default */

  pm3fb_common_init(l_fb_info);
}
#endif

#ifdef MODULE
int init_module(void)
{
  DTRACE;
 
  {
    unsigned long i;
    pm3fb_info *l_fb_info;
	for (i = 0; i < PM3_MAX_BOARD; i++)
	  {
		l_fb_info = &(fb_info[i]);
		l_fb_info->vIOBase = (unsigned char*)-1;
		l_fb_info->board_num = -1;
	  }
  }
  
  pm3fb_init();

  return 0;
}

void cleanup_module(void)
{
  DTRACE;
  {
    unsigned long i;
    pm3fb_info *l_fb_info;
    for (i = 0; i < PM3_MAX_BOARD; i++)
      {
		l_fb_info = &(fb_info[i]);
		if (l_fb_info->board_num != -1)
		{
		  if (l_fb_info->vIOBase != (unsigned char*)-1)
			pm3_unmapIO(l_fb_info);
		  unregister_framebuffer(&l_fb_info->gen.info);
      }
  }

  return;
}
#endif /* MODULE */
