/*
 *  linux/drivers/video/pm3fb.c -- 3DLabs Permedia3 frame buffer device
 *  
 *  Copyright (C) 2001 Romain Dolbeau <dolbeau@irisa.fr>
 *  Based on code written by:
 *           Sven Luther, <luther@dpt-info.u-strasbg.fr>
 *           Alan Hourihane, <alanh@fairlite.demon.co.uk>
 *  Based on linux/drivers/video/skeletonfb.c:
 *	Copyright (C) 1997 Geert Uytterhoeven
 *  Based on linux/driver/video/pm2fb.c:
 *      Copyright (C) 1998-1999 Ilario Nardinocchi (nardinoc@CS.UniBO.IT)
 *      Copyright (C) 1999 Jakub Jelinek (jakub@redhat.com)
 *
 *  This file is subject to the terms and conditions of the GNU General Public
 *  License. See the file COPYING in the main directory of this archive for
 *  more details.
 *
 *  $Header: /home/pm3fb/pm3fb/pm3fb.c,v 1.24 2001/03/04 21:20:58 dolbeau Exp $
 *
 *  CHANGELOG:
 *  Sun Mar  4 22:21:50 CET 2001, v 1.1.2: (numerous) bug fixes.
 *  Fri Mar  2 15:54:07 CET 2001, v 1.1.1: Might have Appian J2000 support, resource mangement in 2.4
 *  Wed Feb 28 18:21:35 CET 2001, v 1.1.0: Might have multiple boards support (added, but not yest tested)
 *  Tue Feb 27 17:31:12 CET 2001, v 1.0.6: fixes boot-time mode select, add more default mode
 *  Tue Feb 27 14:01:36 CET 2001, v 1.0.5: fixes (1.0.4 was broken for 2.2), cleaning up
 *  Mon Feb 26 23:17:36 CET 2001, v 1.0.4: preliminary 2.4.x support, dropped (useless on pm3) partial product, more OF fix
 *  Mon Feb 26 20:59:05 CET 2001, v 1.0.3: No more shadow register (and wasted memory), endianess fix, use OF-preset resolution by default
 *  Wed Feb 21 22:09:30 CET 2001, v 1.0.2: Code cleaning for future multiboard support, better OF support, bugs fix
 *  Wed Feb 21 19:58:56 CET 2001, v 1.0.1: OpenFirmware support, fixed memory detection, better debug support, code cleaning
 *  Wed Feb 21 14:47:06 CET 2001, v 1.0.0: First working version
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/mm.h>
#include <linux/tty.h>
#include <linux/malloc.h>
#include <linux/vmalloc.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <asm/uaccess.h>
#include <linux/fb.h>
#include <linux/init.h>
#include <linux/version.h>

#include <video/fbcon.h>
#include <video/fbcon-mfb.h>
#include <video/fbcon-cfb2.h>
#include <video/fbcon-cfb4.h>
#include <video/fbcon-cfb8.h>
#include <video/fbcon-cfb16.h>
#include <video/fbcon-cfb24.h>
#include <video/fbcon-cfb32.h>

#include <asm/io.h>
#include <linux/pci.h>
#include <linux/ioport.h>
#ifdef CONFIG_FB_OF
#include <asm/prom.h>
#endif

#include "pm3fb.h"

/* ************************************* */
/* ***** The various "global" data ***** */
/* ************************************* */

/* those will need a rework for multiple board support */
/* Driver name */
static const char permedia3_name[16]="Permedia3";

/* the fb_par struct, mandatory */
struct pm3fb_par {
  u32 pixclock;		/* pixclock in KHz */
  
  u32 width;		/* width of virtual screen */
  u32 height;		/* height of virtual screen */
  
  u32 hsstart;		/* horiz. sync start */
  u32 hsend;		/* horiz. sync end */
  u32 hbend;		/* horiz. blank end (also gate end) */
  u32 htotal;		/* total width (w/ sync & blank) */
  
  u32 vsstart;		/* vert. sync start */
  u32 vsend;		/* vert. sync end */
  u32 vbend;		/* vert. blank end */
  u32 vtotal;		/* total height (w/ sync & blank) */
  
  u32 stride;		/* screen stride */
  u32 base;		/* screen base (xoffset+yoffset) */
  u32 depth;		/* screen depth (8, 16, 24 or 32) */
  u32 video;		/* video control (hsync,vsync) */
 };

/* the fb_info struct, mandatory */
struct pm3fb_info
{
  struct fb_info_gen gen;
  unsigned long board_num;
  unsigned long use_current;
  struct pm3fb_par *current_par;
  struct pci_dev *dev;
#ifdef SUPPORT_FB_OF
  struct device_node *dn;
#endif /* SUPPORT_FB_OF */
  unsigned char* fb_base;	/* framebuffer memory base */
  u32 fb_size;		/* framebuffer memory size */
  unsigned char* p_fb;	/* physical address of frame buffer */
  unsigned char* v_fb;	/* virtual address of frame buffer */
  unsigned char *pIOBase;/* physical address of registers region, must be rg_base or rg_base+PM2_REGS_SIZE depending on the host endianness */
  unsigned char *vIOBase; /* address of registers after ioremap() */
  unsigned char BppShift;
  struct {
	u8 transp;
	u8 red;
	u8 green;
	u8 blue;
  } palette[256];
	union {
#ifdef FBCON_HAS_CFB16
		u16 cmap16[16];
#endif
#ifdef FBCON_HAS_CFB32
		u32 cmap32[16];
#endif
	} cmap;
};

/* regular resolution database*/
static struct
{
  char name[16];
  struct pm3fb_par user_mode;
} mode_base[] __initdata =
{
  {"default-800x600", {49500,800,600,16,96,256,1056,1,4,25,625,800,0,8,
		  PM3VideoControl_ENABLE|
		  PM3VideoControl_HSYNC_ACTIVE_HIGH|
		  PM3VideoControl_VSYNC_ACTIVE_HIGH|
		  PM3VideoControl_PIXELSIZE_8BIT}},
  {"1024x768-74-32", {78752,1024,768,32,128,304,1328,1,4,38,806,1024,0,32,
		   PM3VideoControl_ENABLE|
		   PM3VideoControl_HSYNC_ACTIVE_HIGH|
		   PM3VideoControl_VSYNC_ACTIVE_HIGH|
		   PM3VideoControl_PIXELSIZE_32BIT}},
/* ##### auto-generated mode, by fbtimings2pm3 */
/* Generated mode : "640x480-60" */
  {"640x480-60", {25174, 640, 480, 16, 112, 160, 800, 10, 12, 45, 525, 640, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_LOW|PM3VideoControl_VSYNC_ACTIVE_LOW|PM3VideoControl_PIXELSIZE_8BIT}}, 
/* Generated mode : "640x480-72" */
  {"640x480-72", {31199, 640, 480, 24, 64, 192, 832, 9, 12, 40, 520, 640, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_LOW|PM3VideoControl_VSYNC_ACTIVE_LOW|PM3VideoControl_PIXELSIZE_8BIT}}, 
/* Generated mode : "640x480-75" */
  {"640x480-75", {31499, 640, 480, 16, 80, 200, 840, 1, 4, 20, 500, 640, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_LOW|PM3VideoControl_VSYNC_ACTIVE_LOW|PM3VideoControl_PIXELSIZE_8BIT}}, 
/* Generated mode : "640x480-90" */
  {"640x480-90", {39909, 640, 480, 32, 72, 192, 832, 25, 39, 53, 533, 640, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_LOW|PM3VideoControl_VSYNC_ACTIVE_LOW|PM3VideoControl_PIXELSIZE_8BIT}}, 
/* Generated mode : "640x480-100" */
  {"640x480-100", {44899, 640, 480, 32, 160, 208, 848, 22, 34, 51, 531, 640, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_LOW|PM3VideoControl_VSYNC_ACTIVE_LOW|PM3VideoControl_PIXELSIZE_8BIT}}, 
/* Generated mode : "800x600-48-lace" */
/* INTERLACED NOT SUPPORTED
  {"800x600-48-lace", {35999, 800, 600, 80, 208, 264, 1064, 11, 23, 102, 702, 800, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_HIGH|PM3VideoControl_VSYNC_ACTIVE_HIGH|PM3VideoControl_PIXELSIZE_8BIT}}, 
   INTERLACED NOT SUPPORTED */
/* Generated mode : "800x600-56" */
  {"800x600-56", {35999, 800, 600, 24, 96, 224, 1024, 1, 3, 25, 625, 800, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_HIGH|PM3VideoControl_VSYNC_ACTIVE_HIGH|PM3VideoControl_PIXELSIZE_8BIT}}, 
/* Generated mode : "800x600-60" */
  {"800x600-60", {40000, 800, 600, 40, 168, 256, 1056, 1, 5, 28, 628, 800, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_HIGH|PM3VideoControl_VSYNC_ACTIVE_HIGH|PM3VideoControl_PIXELSIZE_8BIT}}, 
/* Generated mode : "800x600-70" */
  {"800x600-70", {44899, 800, 600, 24, 168, 208, 1008, 9, 21, 36, 636, 800, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_HIGH|PM3VideoControl_VSYNC_ACTIVE_LOW|PM3VideoControl_PIXELSIZE_8BIT}}, 
/* Generated mode : "800x600-72" */
  {"800x600-72", {50000, 800, 600, 56, 176, 240, 1040, 37, 43, 66, 666, 800, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_HIGH|PM3VideoControl_VSYNC_ACTIVE_HIGH|PM3VideoControl_PIXELSIZE_8BIT}}, 
/* Generated mode : "800x600-75" */
  {"800x600-75", {49497, 800, 600, 16, 96, 256, 1056, 1, 4, 25, 625, 800, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_HIGH|PM3VideoControl_VSYNC_ACTIVE_HIGH|PM3VideoControl_PIXELSIZE_8BIT}}, 
/* Generated mode : "800x600-90" */
  {"800x600-90", {56637, 800, 600, 8, 72, 192, 992, 8, 19, 35, 635, 800, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_HIGH|PM3VideoControl_VSYNC_ACTIVE_HIGH|PM3VideoControl_PIXELSIZE_8BIT}}, 
/* Generated mode : "800x600-100" */
  {"800x600-100", {67499, 800, 600, 0, 64, 280, 1080, 7, 11, 25, 625, 800, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_HIGH|PM3VideoControl_VSYNC_ACTIVE_HIGH|PM3VideoControl_PIXELSIZE_8BIT}}, 
/* Generated mode : "1024x768-43-lace" */
/* INTERLACED NOT SUPPORTED
  {"1024x768-43-lace", {44899, 1024, 768, 8, 184, 240, 1264, 1, 9, 49, 817, 1024, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_HIGH|PM3VideoControl_VSYNC_ACTIVE_HIGH|PM3VideoControl_PIXELSIZE_8BIT}}, 
   INTERLACED NOT SUPPORTED */
/* Generated mode : "1024x768-60" */
  {"1024x768-60", {64998, 1024, 768, 24, 160, 320, 1344, 3, 9, 38, 806, 1024, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_LOW|PM3VideoControl_VSYNC_ACTIVE_LOW|PM3VideoControl_PIXELSIZE_8BIT}}, 
/* Generated mode : "1024x768-70" */
  {"1024x768-70", {74996, 1024, 768, 24, 160, 304, 1328, 3, 9, 38, 806, 1024, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_LOW|PM3VideoControl_VSYNC_ACTIVE_LOW|PM3VideoControl_PIXELSIZE_8BIT}}, 
/* Generated mode : "1024x768-72" */
  {"1024x768-72", {74996, 10224, 768, 24, 160, 264, 10488, 3, 9, 38, 806, 10224, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_LOW|PM3VideoControl_VSYNC_ACTIVE_LOW|PM3VideoControl_PIXELSIZE_8BIT}}, 
/* Generated mode : "1024x768-75" */
  {"1024x768-75", {78746, 1024, 768, 16, 112, 288, 1312, 1, 4, 32, 800, 1024, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_HIGH|PM3VideoControl_VSYNC_ACTIVE_HIGH|PM3VideoControl_PIXELSIZE_8BIT}}, 
/* Generated mode : "1024x768-90" */
  {"1024x768-90", {100000, 1024, 768, 0, 96, 288, 1312, 21, 36, 77, 845, 1024, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_LOW|PM3VideoControl_VSYNC_ACTIVE_LOW|PM3VideoControl_PIXELSIZE_8BIT}}, 
/* Generated mode : "1024x768-100" */
  {"1024x768-100", {109998, 1024, 768, 0, 88, 368, 1392, 0, 8, 24, 792, 1024, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_LOW|PM3VideoControl_VSYNC_ACTIVE_LOW|PM3VideoControl_PIXELSIZE_8BIT}}, 
/* Generated mode : "1152x864-43-lace" */
/* INTERLACED NOT SUPPORTED
  {"1152x864-43-lace", {64998, 1152, 864, 72, 200, 264, 1416, 78, 87, 191, 1055, 1152, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_HIGH|PM3VideoControl_VSYNC_ACTIVE_HIGH|PM3VideoControl_PIXELSIZE_8BIT}}, 
   INTERLACED NOT SUPPORTED */
/* Generated mode : "1152x864-47-lace" */
/* INTERLACED NOT SUPPORTED
  {"1152x864-47-lace", {64998, 1152, 864, 88, 216, 296, 1448, 30, 39, 83, 947, 1152, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_HIGH|PM3VideoControl_VSYNC_ACTIVE_HIGH|PM3VideoControl_PIXELSIZE_8BIT}}, 
   INTERLACED NOT SUPPORTED */
/* Generated mode : "1152x864-60" */
  {"1152x864-60", {80000, 1152, 864, 64, 176, 304, 1456, 6, 11, 52, 916, 1152, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_HIGH|PM3VideoControl_VSYNC_ACTIVE_HIGH|PM3VideoControl_PIXELSIZE_8BIT}}, 
/* Generated mode : "1152x864-70" */
  {"1152x864-70", {100000, 1152, 864, 40, 192, 360, 1512, 13, 24, 81, 945, 1152, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_HIGH|PM3VideoControl_VSYNC_ACTIVE_HIGH|PM3VideoControl_PIXELSIZE_8BIT}}, 
/* Generated mode : "1152x864-75" */
  {"1152x864-75", {109998, 1152, 864, 24, 168, 312, 1464, 45, 53, 138, 1002, 1152, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_HIGH|PM3VideoControl_VSYNC_ACTIVE_HIGH|PM3VideoControl_PIXELSIZE_8BIT}}, 
/* Generated mode : "1152x864-80" */
  {"1152x864-80", {109998, 1152, 864, 16, 128, 288, 1440, 30, 37, 94, 958, 1152, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_HIGH|PM3VideoControl_VSYNC_ACTIVE_HIGH|PM3VideoControl_PIXELSIZE_8BIT}}, 
/* Generated mode : "1280x1024-43-lace" */
/* INTERLACED NOT SUPPORTED
  {"1280x1024-43-lace", {80000, 1024, 1024, 80, 160, 320, 1344, 50, 60, 125, 1149, 1024, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_HIGH|PM3VideoControl_VSYNC_ACTIVE_HIGH|PM3VideoControl_PIXELSIZE_8BIT}}, 
   INTERLACED NOT SUPPORTED */
/* Generated mode : "1280x1024-47-lace" */
/* INTERLACED NOT SUPPORTED
  {"1280x1024-47-lace", {80000, 1280, 1024, 80, 160, 320, 1600, 1, 11, 29, 1053, 1280, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_HIGH|PM3VideoControl_VSYNC_ACTIVE_HIGH|PM3VideoControl_PIXELSIZE_8BIT}}, 
   INTERLACED NOT SUPPORTED */
/* Generated mode : "1280x1024-60" */
  {"1280x1024-60", {107991, 1280, 1024, 48, 160, 408, 1688, 1, 4, 42, 1066, 1280, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_HIGH|PM3VideoControl_VSYNC_ACTIVE_HIGH|PM3VideoControl_PIXELSIZE_8BIT}}, 
/* Generated mode : "1280x1024-70" */
  {"1280x1024-70", {125992, 1280, 1024, 80, 192, 408, 1688, 1, 6, 42, 1066, 1280, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_HIGH|PM3VideoControl_VSYNC_ACTIVE_HIGH|PM3VideoControl_PIXELSIZE_8BIT}}, 
/* Generated mode : "1280x1024-74" */
  {"1280x1024-74", {134989, 1280, 1024, 32, 176, 432, 1712, 0, 30, 40, 1064, 1280, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_HIGH|PM3VideoControl_VSYNC_ACTIVE_HIGH|PM3VideoControl_PIXELSIZE_8BIT}}, 
/* Generated mode : "1280x1024-75" */
  {"1280x1024-75", {134989, 1280, 1024, 16, 160, 408, 1688, 1, 4, 42, 1066, 1280, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_HIGH|PM3VideoControl_VSYNC_ACTIVE_HIGH|PM3VideoControl_PIXELSIZE_8BIT}}, 
/* Generated mode : "1600x1200-60" */
  {"1600x1200-60", {155981, 1600, 1200, 32, 192, 448, 2048, 10, 18, 70, 1270, 1600, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_LOW|PM3VideoControl_VSYNC_ACTIVE_LOW|PM3VideoControl_PIXELSIZE_8BIT}}, 
/* Generated mode : "1600x1200-66" */
  {"1600x1200-66", {171998, 1600, 1200, 40, 176, 480, 2080, 3, 6, 53, 1253, 1600, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_LOW|PM3VideoControl_VSYNC_ACTIVE_LOW|PM3VideoControl_PIXELSIZE_8BIT}}, 
/* Generated mode : "1600x1200-76" */
  {"1600x1200-76", {197980, 1600, 1200, 40, 176, 480, 2080, 3, 8, 50, 1250, 1600, 0, 8, PM3VideoControl_ENABLE|PM3VideoControl_HSYNC_ACTIVE_LOW|PM3VideoControl_VSYNC_ACTIVE_LOW|PM3VideoControl_PIXELSIZE_8BIT}}, 
/* ##### end of auto-generated mode */
  {"\0", }
};

/* more mandatory stuff (see skeletonfb.c + framebuffer driver HOWTO */
static struct pm3fb_info fb_info[PM3_MAX_BOARD];
static unsigned long detected_board = 0;
static struct pm3fb_par current_par[PM3_MAX_BOARD];
static int current_par_valid[PM3_MAX_BOARD];
/* to allow explicit filtering of board */
  short bus[PM3_MAX_BOARD];
  short slot[PM3_MAX_BOARD];
  short func[PM3_MAX_BOARD];
  short disable[PM3_MAX_BOARD]; 
static struct display disp[PM3_MAX_BOARD];
static char g_options[PM3_OPTIONS_SIZE] __initdata = "pm3fb,dummy";

/* really, inside display */
/* static struct fb_var_screeninfo default_var; */

static int currcon = 0;
static int inverse = 0;

/* ********************* */
/* ***** prototype ***** */
/* ********************* */
/* card-specific */
static void pm3fb_j2000_setup(struct pm3fb_info *l_fb_info);
/* permedia3-specific */
static unsigned long pm3fb_read_dac_reg(struct pm3fb_info *l_fb_info, unsigned long r);
static unsigned long pm3fb_CalculateClock(struct pm3fb_info *l_fb_info,
					  unsigned long reqclock, /* In kHz units */
					  unsigned long refclock, /* In kHz units */
					  unsigned char *prescale,/* ClkPreScale */
					  unsigned char *feedback,/* ClkFeedBackScale */
					  unsigned char *postscale/* ClkPostScale */);
static void pm3fb_common_init(struct pm3fb_info *l_fb_info);
static int pm3fb_Shiftbpp(struct pm3fb_info *l_fb_info, unsigned long depth, int v);
static int pm3fb_Unshiftbpp(struct pm3fb_info *l_fb_info, unsigned long depth, int v);
static void pm3fb_mapIO(struct pm3fb_info *l_fb_info);
static void pm3fb_unmapIO(struct pm3fb_info *l_fb_info);
static void pm3fb_show_cur_mode(struct pm3fb_info *l_fb_info);
static void  pm3fb_show_cur_timing(struct pm3fb_info *l_fb_info);
static void pm3fb_write_mode(struct pm3fb_info *l_fb_info);
static void pm3fb_read_mode(struct pm3fb_info *l_fb_info, struct pm3fb_par *curpar);
static unsigned long pm3fb_size_memory(struct pm3fb_info *l_fb_info);
static void pm3fb_common_init(struct pm3fb_info *l_fb_info);
/* pre-init */
static void pm3fb_mode_setup(char *mode, unsigned long board_num);
static void pm3fb_pciid_setup(char *pciid, unsigned long board_num);
static char* pm3fb_boardnum_setup(char *options, unsigned long *bn);
static void pm3fb_real_setup(char *options);
/* fbdev */
static int pm3fb_open(struct fb_info* info, int user);
static int pm3fb_release(struct fb_info* info, int user);
static int pm3fb_encode_fix(struct fb_fix_screeninfo *fix,
			    const void *par,
			    struct fb_info_gen *info);
static int pm3fb_decode_var(const struct fb_var_screeninfo* var,
			    void *par,
			    struct fb_info_gen* info);
static int pm3fb_encode_var(struct fb_var_screeninfo *var,
			    const void *par,
			    struct fb_info_gen *info);
static void pm3fb_get_par(void *par, struct fb_info_gen *info);
static void pm3fb_set_par(const void *par, struct fb_info_gen *info);
static void pm3fb_set_color(struct pm3fb_info* l_fb_info, unsigned char regno,
			    unsigned char r, unsigned char g, unsigned char b);
static int pm3fb_getcolreg(unsigned regno, unsigned *red, unsigned *green,
			   unsigned *blue, unsigned *transp,
			   struct fb_info *info);
static int pm3fb_setcolreg(unsigned regno, unsigned red, unsigned green,
			   unsigned blue, unsigned transp,
			   struct fb_info *info);
static int pm3fb_blank(int blank_mode, struct fb_info_gen *info);
static void pm3fb_set_disp(const void *par, struct display *disp,
			   struct fb_info_gen *info);
static void pm3fb_detect(void);
/* the struct that hold them together */
#ifdef KERNEL_2_2
struct fbgen_hwswitch pm3fb_switch = {
    pm3fb_detect, pm3fb_encode_fix, pm3fb_decode_var, pm3fb_encode_var,
    pm3fb_get_par, pm3fb_set_par, pm3fb_getcolreg, pm3fb_setcolreg,
    fbgen_pan_display, pm3fb_blank, pm3fb_set_disp
};

static struct fb_ops pm3fb_ops = {
    pm3fb_open, pm3fb_release,
    fbgen_get_fix, fbgen_get_var, fbgen_set_var,
    fbgen_get_cmap, fbgen_set_cmap, fbgen_pan_display, fbgen_ioctl, NULL, NULL
};
#endif /* KERNEL_2_2 */
#ifdef KERNEL_2_4
struct fbgen_hwswitch pm3fb_switch = {
    pm3fb_detect, pm3fb_encode_fix, pm3fb_decode_var, pm3fb_encode_var,
    pm3fb_get_par, pm3fb_set_par, pm3fb_getcolreg, pm3fb_setcolreg,
    fbgen_pan_display, pm3fb_blank, pm3fb_set_disp
};

static struct fb_ops pm3fb_ops = {
    THIS_MODULE,
    pm3fb_open, pm3fb_release,
    fbgen_get_fix, fbgen_get_var, fbgen_set_var,
    fbgen_get_cmap, fbgen_set_cmap, fbgen_pan_display, NULL, NULL, NULL
};
#endif /* KERNEL_2_4 */

/* ********************************** */
/* ***** card-specific function ***** */
/* ********************************** */
static void pm3fb_j2000_setup(struct pm3fb_info *l_fb_info)
{ /* the appian j2000 require more initialization of the second head */
  /* l_fb_info must point to the _second_ head of the J2000 */
  unsigned char m,n,p;
  unsigned long clockused;

  DTRACE;

  /* Memory timings for the Appian J2000 board. */
  PM3_SLOW_WRITE_REG(0x02e311B8, PM3LocalMemCaps);
  PM3_SLOW_WRITE_REG(0x07424905, PM3LocalMemTimings);
  PM3_SLOW_WRITE_REG(0x0c000003, PM3LocalMemControl);
  PM3_SLOW_WRITE_REG(0x00000061, PM3LocalMemRefresh);
  PM3_SLOW_WRITE_REG(0x00000000, PM3LocalMemPowerDown);

  clockused = pm3fb_CalculateClock(l_fb_info, 2*105000, PM3_REF_CLOCK, &m,&n,&p);

  PM3_WRITE_DAC_REG(PM3RD_KClkPreScale, m);
  PM3_WRITE_DAC_REG(PM3RD_KClkFeedbackScale, n);
  PM3_WRITE_DAC_REG(PM3RD_KClkPostScale, p);
  PM3_WRITE_DAC_REG(PM3RD_KClkControl,
					PM3RD_KClkControl_STATE_RUN |
					PM3RD_KClkControl_SOURCE_PLL |
					PM3RD_KClkControl_ENABLE);
  PM3_WRITE_DAC_REG(PM3RD_MClkControl,
					PM3RD_MClkControl_STATE_RUN |
					PM3RD_MClkControl_SOURCE_KCLK |
					PM3RD_MClkControl_ENABLE);
  PM3_WRITE_DAC_REG(PM3RD_SClkControl,
					PM3RD_SClkControl_STATE_RUN |
					PM3RD_SClkControl_SOURCE_PCLK |
					PM3RD_SClkControl_ENABLE);
}

/* *************************************** */
/* ***** permedia3-specific function ***** */
/* *************************************** */

static unsigned long pm3fb_read_dac_reg(struct pm3fb_info *l_fb_info, unsigned long r)
{
  DASSERT((l_fb_info->vIOBase != (unsigned char*)(-1)), "l_fb_info->vIOBase mapped in read dac reg\n"); \
  PM3_SET_INDEX(r);
  mb();
  return(PM3_READ_REG(PM3RD_IndexedData));
}

/* Calculating various clock parameter */
static unsigned long pm3fb_CalculateClock(struct pm3fb_info *l_fb_info,
					  unsigned long reqclock, /* In kHz units */
					  unsigned long refclock, /* In kHz units */
					  unsigned char *prescale,/* ClkPreScale */
					  unsigned char *feedback,/* ClkFeedBackScale */
					  unsigned char *postscale/* ClkPostScale */)
{
    int			f, pre, post;
    unsigned long	freq;
    long		freqerr = 1000;
    unsigned long  	actualclock = 0;

    DTRACE;

    for (f=1;f<256;f++)
      {
	for (pre=1;pre<256;pre++)
	  {
	    for (post=0;post<5;post++)
	      { 
		freq = ((2* refclock * f) / (pre * (1 << post)));
		if ((reqclock > freq - freqerr)&&(reqclock < freq + freqerr))
		  {
		    freqerr = (reqclock > freq) ? 
		      reqclock - freq : freq - reqclock;
		    *feedback = f;
		    *prescale = pre;
		    *postscale = post;
		    actualclock = freq;
		  }
	      }
	  }
      }
    
    return(actualclock);
}

static int pm3fb_Shiftbpp(struct pm3fb_info *l_fb_info, unsigned long depth, int v)
{
  DTRACE;

  switch (depth)
    {
    case 8:
      l_fb_info->BppShift = 4;
      return(v >> 4);
    case 16:
      l_fb_info->BppShift = 3;
      return(v >> 3);
    case 32:
      l_fb_info->BppShift = 2;
      return(v >> 2);
    }
  DPRINTK(1, "Unsupported depth %ld\n", depth);
  return(0);
}

static int pm3fb_Unshiftbpp(struct pm3fb_info *l_fb_info, unsigned long depth, int v)
{
  DTRACE;

  switch (depth)
    {
    case 8:
      l_fb_info->BppShift = 4;
      return(v << 4);
    case 16:
      l_fb_info->BppShift = 3;
      return(v << 3);
    case 32:
      l_fb_info->BppShift = 2;
      return(v << 2);
    }
  DPRINTK(1, "Unsupported depth %ld\n", depth);
  return(0);
}

static void pm3fb_mapIO(struct pm3fb_info *l_fb_info)
{
  DTRACE;

  l_fb_info->vIOBase = ioremap((unsigned long)l_fb_info->pIOBase, 0x20000);
  l_fb_info->v_fb = ioremap((unsigned long)l_fb_info->p_fb, l_fb_info->fb_size);
  DPRINTK(2, "IO mapping : IOBase %lx / %lx, fb %lx / %lx\n",
	  (unsigned long)l_fb_info->pIOBase,
	  (unsigned long)l_fb_info->vIOBase,
	  (unsigned long)l_fb_info->p_fb,
	  (unsigned long)l_fb_info->v_fb);
}

static void pm3fb_unmapIO(struct pm3fb_info *l_fb_info)
{
  DTRACE;

  iounmap(l_fb_info->vIOBase);
  iounmap(l_fb_info->v_fb);
  l_fb_info->vIOBase = (unsigned char *)-1;
  l_fb_info->v_fb = (unsigned char *)-1;
}

static void pm3fb_show_cur_mode(struct pm3fb_info *l_fb_info)
{
  DPRINTK(2, "PM3Aperture0: 0x%08x\n", PM3_READ_REG(PM3Aperture0));
  DPRINTK(2, "PM3Aperture1: 0x%08x\n", PM3_READ_REG(PM3Aperture1));
  DPRINTK(2, "PM3ByAperture1Mode: 0x%08x\n", PM3_READ_REG(PM3ByAperture1Mode));
  DPRINTK(2, "PM3ByAperture2Mode: 0x%08x\n", PM3_READ_REG(PM3ByAperture2Mode));
  DPRINTK(2, "PM3ChipConfig: 0x%08x\n", PM3_READ_REG(PM3ChipConfig));
  DPRINTK(2, "PM3FIFODis: 0x%08x\n", PM3_READ_REG(PM3FIFODis));
  DPRINTK(2, "PM3HTotal: 0x%08x\n", PM3_READ_REG(PM3HTotal));
  DPRINTK(2, "PM3HbEnd: 0x%08x\n", PM3_READ_REG(PM3HbEnd));
  DPRINTK(2, "PM3HgEnd: 0x%08x\n", PM3_READ_REG(PM3HgEnd));
  DPRINTK(2, "PM3HsEnd: 0x%08x\n", PM3_READ_REG(PM3HsEnd));
  DPRINTK(2, "PM3HsStart: 0x%08x\n", PM3_READ_REG(PM3HsStart));
  DPRINTK(2, "PM3MemBypassWriteMask: 0x%08x\n", PM3_READ_REG(PM3MemBypassWriteMask));
  DPRINTK(2, "PM3RD_IndexControl: 0x%08x\n", PM3_READ_REG(PM3RD_IndexControl));
  DPRINTK(2, "PM3ScreenBase: 0x%08x\n", PM3_READ_REG(PM3ScreenBase));
  DPRINTK(2, "PM3ScreenStride: 0x%08x\n", PM3_READ_REG(PM3ScreenStride));
  DPRINTK(2, "PM3VClkCtl: 0x%08x\n", PM3_READ_REG(PM3VClkCtl));
  DPRINTK(2, "PM3VTotal: 0x%08x\n", PM3_READ_REG(PM3VTotal));
  DPRINTK(2, "PM3VbEnd: 0x%08x\n", PM3_READ_REG(PM3VbEnd));
  DPRINTK(2, "PM3VideoControl: 0x%08x\n", PM3_READ_REG(PM3VideoControl));
  DPRINTK(2, "PM3VsEnd: 0x%08x\n", PM3_READ_REG(PM3VsEnd));
  DPRINTK(2, "PM3VsStart: 0x%08x\n", PM3_READ_REG(PM3VsStart));

  DPRINTK(2, "PM3RD_ColorFormat: %ld\n", PM3_READ_DAC_REG(PM3RD_ColorFormat));
  DPRINTK(2, "PM3RD_DACControl: %ld\n", PM3_READ_DAC_REG(PM3RD_DACControl));
  DPRINTK(2, "PM3RD_DClk0FeedbackScale: %ld\n", PM3_READ_DAC_REG(PM3RD_DClk0FeedbackScale));
  DPRINTK(2, "PM3RD_DClk0PostScale: %ld\n", PM3_READ_DAC_REG(PM3RD_DClk0PostScale));
  DPRINTK(2, "PM3RD_DClk0PreScale: %ld\n", PM3_READ_DAC_REG(PM3RD_DClk0PreScale));
  DPRINTK(2, "[not set] PM3RD_IndexControl: %ld\n", PM3_READ_DAC_REG(PM3RD_IndexControl));
  DPRINTK(2, "PM3RD_MiscControl: %ld\n", PM3_READ_DAC_REG(PM3RD_MiscControl));
  DPRINTK(2, "PM3RD_PixelSize: %ld\n", PM3_READ_DAC_REG(PM3RD_PixelSize));
  DPRINTK(2, "PM3RD_SyncControl: %ld\n", PM3_READ_DAC_REG(PM3RD_SyncControl));
}

static void  pm3fb_show_cur_timing(struct pm3fb_info *l_fb_info)
{
  DPRINTK(2, "PM3LocalMemCaps: 0x%08x\n", PM3_READ_REG(PM3LocalMemCaps));
  DPRINTK(2, "PM3LocalMemTimings: 0x%08x\n", PM3_READ_REG(PM3LocalMemTimings));
  DPRINTK(2, "PM3LocalMemControl: 0x%08x\n", PM3_READ_REG(PM3LocalMemControl));
  DPRINTK(2, "PM3LocalMemRefresh: 0x%08x\n", PM3_READ_REG(PM3LocalMemRefresh));
  DPRINTK(2, "PM3LocalMemPowerDown: 0x%08x\n", PM3_READ_REG(PM3LocalMemPowerDown));
}

/* write the mode to registers */
static void pm3fb_write_mode(struct pm3fb_info *l_fb_info)
{
  DTRACE;

  PM3_SLOW_WRITE_REG(PM3MemBypassWriteMask, 0xffffffff);
  PM3_SLOW_WRITE_REG(PM3Aperture0, 0x00000000);
  PM3_SLOW_WRITE_REG(PM3Aperture1, 0x00000000);
  PM3_SLOW_WRITE_REG(PM3FIFODis,  0x00000007);

  PM3_SLOW_WRITE_REG(PM3HTotal,
		     pm3fb_Shiftbpp(l_fb_info, l_fb_info->current_par->depth,
				    l_fb_info->current_par->htotal - 1));
  PM3_SLOW_WRITE_REG(PM3HsEnd,
		     pm3fb_Shiftbpp(l_fb_info, l_fb_info->current_par->depth,
				    l_fb_info->current_par->hsend));
  PM3_SLOW_WRITE_REG(PM3HsStart,
		     pm3fb_Shiftbpp(l_fb_info, l_fb_info->current_par->depth,
				    l_fb_info->current_par->hsstart));
  PM3_SLOW_WRITE_REG(PM3HbEnd,
		     pm3fb_Shiftbpp(l_fb_info, l_fb_info->current_par->depth,
				    l_fb_info->current_par->hbend));
  PM3_SLOW_WRITE_REG(PM3HgEnd,
		     pm3fb_Shiftbpp(l_fb_info, l_fb_info->current_par->depth,
				    l_fb_info->current_par->hbend));
  PM3_SLOW_WRITE_REG(PM3ScreenStride,
		     pm3fb_Shiftbpp(l_fb_info, l_fb_info->current_par->depth,
				    l_fb_info->current_par->stride));
  PM3_SLOW_WRITE_REG(PM3VTotal, l_fb_info->current_par->vtotal - 1);
  PM3_SLOW_WRITE_REG(PM3VsEnd, l_fb_info->current_par->vsend - 1);
  PM3_SLOW_WRITE_REG(PM3VsStart, l_fb_info->current_par->vsstart - 1);
  PM3_SLOW_WRITE_REG(PM3VbEnd, l_fb_info->current_par->vbend);

  switch (l_fb_info->current_par->depth)
    {
    case 8:
      PM3_SLOW_WRITE_REG(PM3ByAperture1Mode, PM3ByApertureMode_PIXELSIZE_8BIT);
      PM3_SLOW_WRITE_REG(PM3ByAperture2Mode, PM3ByApertureMode_PIXELSIZE_8BIT);
      break;

    case 16:
#ifndef __BIG_ENDIAN
      PM3_SLOW_WRITE_REG(PM3ByAperture1Mode, PM3ByApertureMode_PIXELSIZE_16BIT);
      PM3_SLOW_WRITE_REG(PM3ByAperture2Mode, PM3ByApertureMode_PIXELSIZE_16BIT);
#else
      PM3_SLOW_WRITE_REG(PM3ByAperture1Mode, PM3ByApertureMode_PIXELSIZE_16BIT |
					PM3ByApertureMode_BYTESWAP_BADC);
      PM3_SLOW_WRITE_REG(PM3ByAperture2Mode, PM3ByApertureMode_PIXELSIZE_16BIT |
					PM3ByApertureMode_BYTESWAP_BADC);
#endif /* ! __BIG_ENDIAN */
      break;

    case 32:
#ifndef __BIG_ENDIAN
      PM3_SLOW_WRITE_REG(PM3ByAperture1Mode, PM3ByApertureMode_PIXELSIZE_32BIT);
      PM3_SLOW_WRITE_REG(PM3ByAperture2Mode, PM3ByApertureMode_PIXELSIZE_32BIT);
#else
      PM3_SLOW_WRITE_REG(PM3ByAperture1Mode, PM3ByApertureMode_PIXELSIZE_32BIT |
		    PM3ByApertureMode_BYTESWAP_DCBA);
      PM3_SLOW_WRITE_REG(PM3ByAperture2Mode, PM3ByApertureMode_PIXELSIZE_32BIT |
		    PM3ByApertureMode_BYTESWAP_DCBA);
#endif /* ! __BIG_ENDIAN */
      break;

	default:
	  DPRINTK(1, "Unsupported depth %ld\n", l_fb_info->current_par->depth);
	  break;
    }

  PM3_SLOW_WRITE_REG(PM3VideoControl, l_fb_info->current_par->video);
  PM3_SLOW_WRITE_REG(PM3VClkCtl, (PM3_READ_REG(PM3VClkCtl) & 0xFFFFFFFC));
  PM3_SLOW_WRITE_REG(PM3ScreenBase, l_fb_info->current_par->base);
  PM3_SLOW_WRITE_REG(PM3ChipConfig, (PM3_READ_REG(PM3ChipConfig) & 0xFFFFFFFD));
  
  {
    unsigned char m;		/* ClkPreScale */
    unsigned char n; 		/* ClkFeedBackScale */
    unsigned char p;		/* ClkPostScale */
    unsigned long usedclock = pm3fb_CalculateClock(l_fb_info, l_fb_info->current_par->pixclock, PM3_REF_CLOCK, &m, &n, &p); /* usedclock unused */

    DPRINTK(2, "Pixclock: %d, Pre: %d, Feedback: %d, Post: %d\n",
	    l_fb_info->current_par->pixclock,
	    (int)m, (int)n, (int)p);

    PM3_WRITE_DAC_REG(PM3RD_DClk0PreScale, m);
    PM3_WRITE_DAC_REG(PM3RD_DClk0FeedbackScale, n);
    PM3_WRITE_DAC_REG(PM3RD_DClk0PostScale, p);
  }
  /*
  PM3_WRITE_DAC_REG(PM3RD_IndexControl, 0x00);
  */
  /*
  PM3_SLOW_WRITE_REG(PM3RD_IndexControl, 0x00);
  */
  {
    char tempsync = 0x00;
    
    if ((l_fb_info->current_par->video & PM3VideoControl_HSYNC_MASK) == PM3VideoControl_HSYNC_ACTIVE_HIGH)
      tempsync |= PM3RD_SyncControl_HSYNC_ACTIVE_HIGH;
    if ((l_fb_info->current_par->video & PM3VideoControl_VSYNC_MASK) == PM3VideoControl_VSYNC_ACTIVE_HIGH)
      tempsync |= PM3RD_SyncControl_VSYNC_ACTIVE_HIGH;
    
    PM3_WRITE_DAC_REG(PM3RD_SyncControl, tempsync);
    DPRINTK(2, "PM3RD_SyncControl: %d\n", tempsync);
  }
  PM3_WRITE_DAC_REG(PM3RD_DACControl, 0x00);
  
  switch (l_fb_info->current_par->depth)
    {
    case 8:
      PM3_WRITE_DAC_REG(PM3RD_PixelSize, PM3RD_PixelSize_8_BIT_PIXELS);
      PM3_WRITE_DAC_REG(PM3RD_ColorFormat,
						PM3RD_ColorFormat_CI8_COLOR|
						PM3RD_ColorFormat_COLOR_ORDER_BLUE_LOW);
      break;
    case 16:
      PM3_WRITE_DAC_REG(PM3RD_PixelSize, PM3RD_PixelSize_16_BIT_PIXELS);
      PM3_WRITE_DAC_REG(PM3RD_ColorFormat,
						PM3RD_ColorFormat_565_FRONT_COLOR|
						PM3RD_ColorFormat_COLOR_ORDER_BLUE_LOW|
						PM3RD_ColorFormat_LINEAR_COLOR_EXT_ENABLE);
      break;
    case 32:
      PM3_WRITE_DAC_REG(PM3RD_PixelSize, PM3RD_PixelSize_32_BIT_PIXELS);
      PM3_WRITE_DAC_REG(PM3RD_ColorFormat,
						PM3RD_ColorFormat_8888_COLOR|
						PM3RD_ColorFormat_COLOR_ORDER_BLUE_LOW);
      break;
    }

  PM3_WRITE_DAC_REG(PM3RD_MiscControl, PM3RD_MiscControl_HIGHCOLOR_RES_ENABLE);

  PM3_SHOW_CUR_MODE;
  PM3_SHOW_CUR_TIMING;
}

static void pm3fb_read_mode(struct pm3fb_info *l_fb_info, struct pm3fb_par *curpar)
{
  unsigned long pixsize1, pixsize2, clockused;
  unsigned long pre, feedback, post;

  DTRACE;

  clockused = PM3_READ_REG(PM3VClkCtl);

  switch (clockused)
    {
    case 3:
      pre = PM3_READ_DAC_REG(PM3RD_DClk3PreScale);
      feedback = PM3_READ_DAC_REG(PM3RD_DClk3FeedbackScale);
      post = PM3_READ_DAC_REG(PM3RD_DClk3PostScale);
      
      DPRINTK(2, "DClk3 parameter: Pre: %ld, Feedback: %ld, Post: %ld ; giving pixclock: %ld\n",
	      pre, feedback, post,
	      PM3_SCALE_TO_CLOCK(pre, feedback, post) );
      break;
    case 2:
      pre = PM3_READ_DAC_REG(PM3RD_DClk2PreScale);
      feedback = PM3_READ_DAC_REG(PM3RD_DClk2FeedbackScale);
      post = PM3_READ_DAC_REG(PM3RD_DClk2PostScale);
      
      DPRINTK(2, "DClk2 parameter: Pre: %ld, Feedback: %ld, Post: %ld ; giving pixclock: %ld\n",
	      pre, feedback, post,
	      PM3_SCALE_TO_CLOCK(pre, feedback, post) );
      break;
    case 1:
      pre = PM3_READ_DAC_REG(PM3RD_DClk1PreScale);
      feedback = PM3_READ_DAC_REG(PM3RD_DClk1FeedbackScale);
      post = PM3_READ_DAC_REG(PM3RD_DClk1PostScale);
      
      DPRINTK(2, "DClk1 parameter: Pre: %ld, Feedback: %ld, Post: %ld ; giving pixclock: %ld\n",
	      pre, feedback, post,
	      PM3_SCALE_TO_CLOCK(pre, feedback, post) );
      break;
    case 0:
      pre = PM3_READ_DAC_REG(PM3RD_DClk0PreScale);
      feedback = PM3_READ_DAC_REG(PM3RD_DClk0FeedbackScale);
      post = PM3_READ_DAC_REG(PM3RD_DClk0PostScale);
      
      DPRINTK(2, "DClk0 parameter: Pre: %ld, Feedback: %ld, Post: %ld ; giving pixclock: %ld\n",
	      pre, feedback, post,
	      PM3_SCALE_TO_CLOCK(pre, feedback, post) );
      break;
    default:
      pre = feedback = post = 0;
      DPRINTK(1, "Unknowk D clock used : %ld\n", clockused);
      break;
    }

  curpar->pixclock = PM3_SCALE_TO_CLOCK(pre, feedback, post);
      
  pixsize1 = PM3ByApertureMode_PIXELSIZE_MASK & (PM3_READ_REG(PM3ByAperture1Mode));
  pixsize2 = PM3ByApertureMode_PIXELSIZE_MASK & (PM3_READ_REG(PM3ByAperture2Mode));

  DASSERT((pixsize1 == pixsize2), "pixsize the same in both aperture\n");

  if (pixsize1 & PM3ByApertureMode_PIXELSIZE_32BIT)
    curpar->depth = 32;
  else
    if (pixsize1 & PM3ByApertureMode_PIXELSIZE_16BIT)
      curpar->depth = 16;
    else
      curpar->depth = 8;

  /* not sure if I need to add one on the next ; it give better result with */
  curpar->htotal = pm3fb_Unshiftbpp(l_fb_info, curpar->depth, 1 + PM3_READ_REG(PM3HTotal));
  curpar->hsend = pm3fb_Unshiftbpp(l_fb_info, curpar->depth, PM3_READ_REG(PM3HsEnd));
  curpar->hsstart = pm3fb_Unshiftbpp(l_fb_info, curpar->depth, PM3_READ_REG(PM3HsStart));
  curpar->hbend = pm3fb_Unshiftbpp(l_fb_info, curpar->depth, PM3_READ_REG(PM3HbEnd));

  curpar->stride = pm3fb_Unshiftbpp(l_fb_info, curpar->depth, PM3_READ_REG(PM3ScreenStride));

  curpar->vtotal = 1 + PM3_READ_REG(PM3VTotal);
  curpar->vsend = 1 + PM3_READ_REG(PM3VsEnd);
  curpar->vsstart = 1 + PM3_READ_REG(PM3VsStart);
  curpar->vbend = PM3_READ_REG(PM3VbEnd);

  curpar->video = PM3_READ_REG(PM3VideoControl);

  curpar->base = 0; /* PM3_READ_REG(PM3ScreenBase); */
  curpar->width = curpar->htotal - curpar->hbend;
  curpar->height = curpar->vtotal - curpar->vbend;

  DPRINTK(2, "Found : %d * %d, %d Khz, stride is %08x\n",
	  curpar->width, curpar->height, curpar->pixclock, curpar->stride);
}

static unsigned long pm3fb_size_memory(struct pm3fb_info *l_fb_info)
{
  unsigned long memsize, tempBypass, i, temp1, temp2;
  
  DTRACE;

  l_fb_info->fb_size = 64*1024*1024; /* pm3 aperture always 64 MB */
  pm3fb_mapIO(l_fb_info); /* temporary map IO */
  
  DASSERT((l_fb_info->vIOBase != NULL), "IO successfully mapped before mem detect\n");
  DASSERT((l_fb_info->v_fb != NULL), "FB successfully mapped before mem detect\n");

  tempBypass = PM3_READ_REG(PM3MemBypassWriteMask);

  DPRINTK(2, "PM3MemBypassWriteMask was: 0x%08lx\n", tempBypass);

  PM3_SLOW_WRITE_REG(PM3MemBypassWriteMask, 0xFFFFFFFF);

  /* pm3 split up memory, replicates, and do a lot of nasty stuff IMHO ;-) */
  for(i = 0; i < 32; i++)
	{
#ifdef MUST_BYTESWAP
	  writel(__swab32(i * 0x00345678), (l_fb_info->v_fb + (i * 1048576)));
#else
	  writel(i * 0x00345678, (l_fb_info->v_fb + (i * 1048576)));
#endif
	  mb();
#ifdef MUST_BYTESWAP
	  temp1 = __swab32(readl((l_fb_info->v_fb + (i * 1048576))));
#else
	  temp1 = readl((l_fb_info->v_fb + (i * 1048576)));
#endif
	  /* Let's check for wrapover, write will fail at 16MB boundary */
	  if (temp1 == (i * 0x00345678)) 
		memsize = i;
	  else 
		break;
	}

  DPRINTK(2, "First detect pass already got %ld MB\n", memsize + 1);

  if (memsize == i)
	{
	  for(i=0;i<32;i++)
		{
		  /* Clear first 32MB ; 0 is 0, no need to byteswap */
		  writel(0x0000000, (l_fb_info->v_fb + (i * 1048576)));
		  mb();
		}

	  for(i=32;i<64;i++)
		{
#ifdef MUST_BYTESWAP
		  writel(__swab32(i * 0x00345678), (l_fb_info->v_fb + (i * 1048576)));
#else
		  writel(i * 0x00345678, (l_fb_info->v_fb + (i * 1048576)));
#endif
		  mb();
#ifdef MUST_BYTESWAP
		  temp1 = __swab32(readl((l_fb_info->v_fb + (i * 1048576))));
		  temp2 = __swab32(readl((l_fb_info->v_fb + ((i - 32) * 1048576))));
#else
		  temp1 = readl((l_fb_info->v_fb + (i * 1048576)));
		  temp2 = readl((l_fb_info->v_fb + ((i - 32) * 1048576)));
#endif
		  if ( (temp1 == (i*0x00345678)) && (temp2 == 0) ) /* different value, different RAM... */
			memsize = i;
		  else 
			break;
		}
	}

  DPRINTK(2, "Second detect pass got %ld MB\n", memsize + 1);

  PM3_SLOW_WRITE_REG(PM3MemBypassWriteMask, tempBypass);

  pm3fb_unmapIO(l_fb_info);
  memsize = 1048576 * (memsize + 1);
  
  DPRINTK(2, "Returning 0x%08lx bytes\n", memsize);

  l_fb_info->fb_size = memsize;

  return(memsize);
}

/* common initialisation */
static void pm3fb_common_init(struct pm3fb_info *l_fb_info)
{
  u16 subvendor, subdevice;

  DTRACE;

  DPRINTK(2, "Initializing board #%ld @ %lx\n", l_fb_info->board_num, (unsigned long)l_fb_info);

  strcpy(l_fb_info->gen.info.modename, permedia3_name);
  disp[l_fb_info->board_num].scrollmode=SCROLL_YNOMOVE;
  l_fb_info->gen.parsize=sizeof(struct pm3fb_par);
  l_fb_info->gen.info.changevar = NULL;
  l_fb_info->gen.info.node = -1;
  l_fb_info->gen.info.fbops = &pm3fb_ops;
  l_fb_info->gen.info.disp = &(disp[l_fb_info->board_num]);
  l_fb_info->gen.info.switch_con = &fbgen_switch;
  l_fb_info->gen.info.updatevar = &fbgen_update_var; /* */
  l_fb_info->gen.info.blank = &fbgen_blank; /* */
  l_fb_info->gen.info.flags = FBINFO_FLAG_DEFAULT;

  DPRINTK(3, "Var filling done\n");

  pm3fb_mapIO(l_fb_info);

  DPRINTK(3, "IO mapping done\n");

  /* card-specific stuff */
  if ((!pci_read_config_word(l_fb_info->dev, PCI_SUBSYSTEM_VENDOR_ID, &subvendor)) &&
      (!pci_read_config_word(l_fb_info->dev, PCI_SUBSYSTEM_ID, &subdevice)))
    {
      /* first, Appian Jeronimo 2000 (dual pm3 + gamma ) */
      if ((subvendor == 0x1097) && /* Appian ? */
	  (subdevice == 0x3d32)) /* J2000 */
	{
	  DPRINTK(1, "Hummm, this is an Appian J2000 board, head number %d\n", PCI_FUNC(l_fb_info->dev->devfn));
	  if ((PCI_FUNC(l_fb_info->dev->devfn)) == 2) /* 2nd Permedia3 */
	    {
	      pm3fb_j2000_setup(l_fb_info);
	    }
	}
    }
  else
    {
      printk("pm3fb: Error: pci_read_config_word failed, board #%d\n", l_fb_info->board_num);
    }
  
  (void)fbgen_get_var(&(disp[l_fb_info->board_num]).var, -1, &l_fb_info->gen.info);
  
  DPRINTK(3, "First fbgen done\n");
  
  (void)fbgen_do_set_var(&(disp[l_fb_info->board_num]).var, 1, &l_fb_info->gen);
  
  DPRINTK(3, "Second fbgen done\n");

  fbgen_set_disp(-1, &l_fb_info->gen);

  DPRINTK(3, "Third fbgen done\n");

  fbgen_install_cmap(0, &l_fb_info->gen);

  DPRINTK(3, "Fourth fbgen done\n");

  if (register_framebuffer(&l_fb_info->gen.info) < 0)
	{
	  DPRINTK(1, "Couldn't register framebuffer\n");
	  return;
	}

  DPRINTK(3, "Register fb done\n");

  PM3_SHOW_CUR_MODE;
  PM3_SHOW_CUR_TIMING;
  
  pm3fb_write_mode(l_fb_info);

  detected_board++;

  printk("fb%d: %s, using %uK of video memory.\n",
	 GET_FB_IDX(l_fb_info->gen.info.node),
	 permedia3_name,
	 (u32)(l_fb_info->fb_size>>10));
}

/* *********************************** */
/* ***** pre-init board(s) setup ***** */
/* *********************************** */

static void pm3fb_mode_setup(char *mode, unsigned long board_num)
{
  struct pm3fb_info *l_fb_info = &(fb_info[board_num]);
  struct pm3fb_par *l_fb_par = &(current_par[board_num]);
  unsigned long i = 0;

  current_par_valid[board_num] = 0;

  if (!strncmp(mode, "current", 7))
	{
	  l_fb_info->use_current = 1; /* default w/ OpenFirmware */
	}
  else
	{
	  while ((mode_base[i].name[0]) && (!current_par_valid[board_num]))
		{
		  if (!(strncmp(mode, mode_base[i].name, strlen(mode_base[i].name))))
		    {
		      memcpy(l_fb_par, &(mode_base[i].user_mode), sizeof(struct pm3fb_par));
		      current_par_valid[board_num] = 1;
		      DPRINTK(2, "Mode set to %s\n", mode_base[i].name);
		    }
		  i++;
		}
	  DASSERT(current_par_valid[board_num], "Valid mode on command line\n");
	}
}

static void pm3fb_pciid_setup(char *pciid, unsigned long board_num)
{
  struct pm3fb_info *l_fb_info = &(fb_info[board_num]);
  short l_bus = -1, l_slot = -1, l_func = -1;
  char *next;

  if (pciid)
    {
      l_bus = simple_strtoul(pciid, &next, 10);
      if (next && (next[0] == ':'))
	{
	  pciid = next + 1;
	  l_slot = simple_strtoul(pciid, &next, 10);
	  if (next && (next[0] == ':'))
	    {
	      pciid = next + 1;
	      l_func = simple_strtoul(pciid, (char**)NULL, 10);
	    }
	}
    }
  
  if ((l_bus >= 0) && (l_slot >= 0) && (l_func >=0))
    {
      bus[board_num] = l_bus;
      slot[board_num] = l_slot;
      func[board_num] = l_func;
      DPRINTK(2, "Board #%ld will be PciId: %hd:%hd:%hd\n", board_num, l_bus, l_slot, l_func);
    }
  else
    {
      DPRINTK(1, "Invalid PciId: %hd:%hd:%hd for board #%ld\n", l_bus, l_slot, l_func, board_num);
    }
}

static char* pm3fb_boardnum_setup(char *options, unsigned long *bn)
{
  char *next;
  
  (*bn) = simple_strtoul(options, &next, 10);

  if (next && (next[0] == ':') && ((*bn) >= 0) && ((*bn) <= PM3_MAX_BOARD))
    {
      return(next + 1);
    }
  else
    {
      (*bn) = 0;
      return(options);
    }
}

static void pm3fb_real_setup(char *options)
{
  char *next;
  unsigned long i, bn;
  struct pm3fb_info *l_fb_info;

  DTRACE;

  DPRINTK(2, "Options : %s\n", options);

  for (i = 0; i < PM3_MAX_BOARD ; i++)
    {
      l_fb_info = &(fb_info[i]);
      memset(l_fb_info, 0, sizeof(struct pm3fb_info));
      l_fb_info->gen.fbhw = &pm3fb_switch;
      l_fb_info->board_num = i;
      current_par_valid[i] = 0;
      slot[i] = -1;
      func[i] = -1;
      bus[i] = -1;
      disable[i] = 0;
      l_fb_info->current_par = &(current_par[i]);
    }
  
  /* eat up prefix pm3fb and whatever is used as separator i.e. :,= */
  if (!strncmp(options, "pm3fb", 5))
    {
      options += 5;
      while (((*options) == ',') || ((*options) == ':') || ((*options) == '='))
	options++;
    }

  while (options)
	{
	  if ((next=strchr(options, ',')))
	    {
	      (*next)='\0';
	      next++;
	    }

	  if (!strncmp(options, "mode:", 5))
	    {
	      options = pm3fb_boardnum_setup(options + 5, &bn);
	      pm3fb_mode_setup(options, bn);
	    } else
	  if (!strncmp(options, "off:", 4))
	    {
	      options = pm3fb_boardnum_setup(options + 4, &bn);		  
	      disable[bn] = 1;
	    } else
	  if (!strncmp(options, "off", 3)) /* to allow for more common video=<driver>,off syntax */
	    {
	      disable[0] = 1;
	    } else
	  if (!strncmp(options, "disable:", 8))
	    {
	      options = pm3fb_boardnum_setup(options + 8, &bn);		  
	      disable[bn] = 1;
	    } else
	  if (!strncmp(options, "pciid:", 6))
	    {
	      options = pm3fb_boardnum_setup(options + 6, &bn);	  
	      pm3fb_pciid_setup(options, bn);
	    }
	  options = next;
	}
}

/* ********************************************** */
/* ***** framebuffer API standard functions ***** */
/* ********************************************** */

static int pm3fb_open(struct fb_info* info, int user)
{
  struct pm3fb_info *l_fb_info = (struct pm3fb_info*) info;

  DTRACE;

  DPRINTK(3, "user is %d\n", user);

  MOD_INC_USE_COUNT;
 
  return(0);
}

static int pm3fb_release(struct fb_info* info, int user)
{
  struct pm3fb_info *l_fb_info = (struct pm3fb_info*) info;

  DTRACE;

  MOD_DEC_USE_COUNT;

  return(0);
}

static int pm3fb_encode_fix(struct fb_fix_screeninfo *fix,
			    const void *par,
			    struct fb_info_gen *info)
{
  struct pm3fb_info* l_fb_info=(struct pm3fb_info* )info;
  struct pm3fb_par* p=(struct pm3fb_par* )par;

  DTRACE;
  
  strcpy(fix->id, permedia3_name);
  fix->smem_start = l_fb_info->p_fb;
  fix->smem_len = l_fb_info->fb_size;
  fix->mmio_start = l_fb_info->pIOBase;
  fix->mmio_len = PM3_REGS_SIZE;
  fix->accel = 0 /* FB_ACCEL_3DLABS_PERMEDIA3 */;
  fix->type = FB_TYPE_PACKED_PIXELS;
  fix->visual= (p->depth == 8) ? FB_VISUAL_PSEUDOCOLOR:FB_VISUAL_TRUECOLOR;
  if (current_par_valid[l_fb_info->board_num])
	fix->line_length = l_fb_info->current_par->width*(l_fb_info->current_par->depth/8);
  else
	fix->line_length = 0;
  fix->xpanstep = 64/p->depth;
  fix->ypanstep = 1;
  fix->ywrapstep = 0;
  return(0);
}

static int pm3fb_decode_var(const struct fb_var_screeninfo* var,
			    void *par,
			    struct fb_info_gen* info)
{
  struct pm3fb_info *l_fb_info = (struct pm3fb_info*)info;
  struct pm3fb_par *p = (struct pm3fb_par*)par;
  struct pm3fb_par temp_p;
  u32 xres;
  
  DTRACE;

  DASSERT((var != NULL), "fb_var_screeninfo* not NULL");
  DASSERT((p != NULL), "pm3fb_par* not NULL");
  DASSERT((l_fb_info != NULL), "pm3fb_info* not NULL");

  memset(&temp_p, 0, sizeof(struct pm3fb_par));
  temp_p.width = (var->xres_virtual + 7) & ~7;
  temp_p.height = var->yres_virtual;
  temp_p.depth = (var->bits_per_pixel + 7) & ~7;
  temp_p.depth = (temp_p.depth > 32) ? 32 : temp_p.depth;

  DPRINTK(2, "xres: %d, yres: %d, vxres: %d, vyres: %d ; xoffset:%d, yoffset: %d\n",
	  var->xres, var->yres, var->xres_virtual, var->yres_virtual, var->xoffset, var->yoffset);

  xres = (var->xres + 31) & ~31;
  if (temp_p.width<xres+var->xoffset)
    temp_p.width=xres+var->xoffset;
  if (temp_p.height<var->yres+var->yoffset)
    temp_p.height=var->yres+var->yoffset;

  DPRINTK(3, "First var filling done\n");
    
  if (temp_p.width>2048) {
    DPRINTK(1, "virtual width not supported: %u\n", temp_p.width);
    return(-EINVAL);
  }
  if (var->yres<200) {
    DPRINTK(1, "height not supported: %u\n",
	    (u32 )var->yres);
    return(-EINVAL);
  }
  if (temp_p.height<200 || temp_p.height>2048) {
    DPRINTK(1, "virtual height not supported: %u\n", temp_p.height);
    return(-EINVAL);
  }
  if (temp_p.depth>32) {
    DPRINTK(1, "depth not supported: %u\n", temp_p.depth);
    return(-EINVAL);
  }
  if (temp_p.width*temp_p.height*temp_p.depth/8>l_fb_info->fb_size) {
    DPRINTK(1, "no memory for screen (%ux%ux%u)\n",
	    temp_p.width, temp_p.height, temp_p.depth);
    return(-EINVAL);
  }

  if ((!var->pixclock) ||
      (!var->right_margin) ||
      (!var->hsync_len) ||
      (!var->left_margin) ||
      (!var->lower_margin) ||
      (!var->vsync_len) ||
      (!var->upper_margin)
      )
    {
      unsigned long i = 0, done = 0;
      printk("pm3fb: refusing to use a likely wrong timing\n");
      
      while ((mode_base[i].user_mode.width) && !done)
	{
	  if ((mode_base[i].user_mode.width == temp_p.width) &&
	      (mode_base[i].user_mode.height == temp_p.height))
	    {
	      printk("pm3fb: using close match %s\n", mode_base[i].name);
	      temp_p = mode_base[i].user_mode;
	      done = 1;
	    }
	  i++;
	}
      if (!done)
	return(-EINVAL);
    }
  else
    {
      temp_p.pixclock=PICOS2KHZ(var->pixclock);
      if (temp_p.pixclock>PM3_MAX_PIXCLOCK)
	{
	  DPRINTK(1, "pixclock too high (%uKHz)\n", temp_p.pixclock);
	  return(-EINVAL);
	}
      
      DPRINTK(3, "Clock done\n");
      
      temp_p.hsstart = var->right_margin;
      temp_p.hsend = var->right_margin + var->hsync_len;
      temp_p.hbend = var->right_margin + var->hsync_len + var->left_margin;
      temp_p.htotal = xres + temp_p.hbend;
      
      temp_p.vsstart = var->lower_margin;
      temp_p.vsend = var->lower_margin + var->vsync_len;
      temp_p.vbend = var->lower_margin + var->vsync_len + var->upper_margin;
      temp_p.vtotal = var->yres + temp_p.vbend;
      
      temp_p.stride = temp_p.width;

      DPRINTK(2, "Using %d * %d, %d Khz, stride is %08x\n",
	      temp_p.width, temp_p.height, temp_p.pixclock, temp_p.stride);
      
      temp_p.base = (var->yoffset * xres) + var->xoffset;
      
      temp_p.video = 0;
      
      if (var->sync & FB_SYNC_HOR_HIGH_ACT)
	temp_p.video |= PM3VideoControl_HSYNC_ACTIVE_HIGH;
      else
	temp_p.video |= PM3VideoControl_HSYNC_ACTIVE_LOW;
      
      if (var->sync & FB_SYNC_VERT_HIGH_ACT)
	temp_p.video |= PM3VideoControl_VSYNC_ACTIVE_HIGH;
      else
	temp_p.video |= PM3VideoControl_VSYNC_ACTIVE_LOW;
      
      if ((var->vmode & FB_VMODE_MASK) == FB_VMODE_INTERLACED)
	{
	  DPRINTK(1, "Interlaced mode not supported\n\n");
	  return(-EINVAL);
	}
      
      if ((var->vmode & FB_VMODE_MASK) == FB_VMODE_DOUBLE)
	temp_p.video |= PM3VideoControl_LINE_DOUBLE_ON;
      else
	temp_p.video |= PM3VideoControl_LINE_DOUBLE_OFF;
      
      if (var->activate == FB_ACTIVATE_NOW)
	temp_p.video |= PM3VideoControl_ENABLE;
      else
	{
	  temp_p.video |= PM3VideoControl_DISABLE;
	  DPRINTK(2, "PM3Video disabled\n");
	}
      
      switch (temp_p.depth)
	{
	case 8:
	  temp_p.video |= PM3VideoControl_PIXELSIZE_8BIT;
	  break;
	case 16:
	  temp_p.video |= PM3VideoControl_PIXELSIZE_16BIT;
	  break;
	case 32:
	  temp_p.video |= PM3VideoControl_PIXELSIZE_32BIT;
	  break;
	default:
	  DPRINTK(1, "Unsupported depth\n");
	  break;
	}
    }
  (*p) = temp_p; 
  
  DPRINTK(3, "All done\n");
  
  return(0);
}

static int pm3fb_encode_var(struct fb_var_screeninfo *var,
			    const void *par,
			    struct fb_info_gen *info)
{
  struct pm3fb_par* p= (struct pm3fb_par*) par;
  struct pm3fb_info *l_fb_info = (struct pm3fb_info*) info;

  u32 base;
  
  DTRACE; 

  DASSERT((var != NULL), "fb_var_screeninfo* not NULL");
  DASSERT((p != NULL), "pm3fb_par* not NULL");
  DASSERT((info != NULL), "fb_info_gen* not NULL");
 
  memset(var, 0, sizeof(struct fb_var_screeninfo));
  /*
    if (p->flags & PM2FF_ACCEL)
    var->accel_flags |= FB_ACCELF_TEXT;
  */
  var->xres_virtual = p->width;
  var->yres_virtual = p->height;
  var->xres = p->htotal - p->hbend;
  var->yres = p->vtotal - p->vbend;

  DPRINTK(2, "xres = %d, yres : %d\n", var->xres, var->yres);

  var->right_margin = p->hsstart;
  var->hsync_len = p->hsend - p->hsstart;
  var->left_margin = p->hbend - p->hsend;
  var->lower_margin = p->vsstart;
  var->vsync_len = p->vsend - p->vsstart;
  var->upper_margin = p->vbend - p->vsend;
  var->bits_per_pixel = p->depth;
  
  switch (p->depth) {
  case 8:
    var->red.length = var->green.length = var->blue.length = 8;
    break;

  case 16:
    var->red.offset = 11;
    var->red.length = 5;
    var->green.offset = 5;
    var->green.length = 6;
    var->blue.length = 5;
    break;

  case 32:
    var->transp.offset = 24;
    var->red.offset = 16;
    var->green.offset = 8;
    var->red.length = var->green.length = var->blue.length = var->transp.length = 8;
    break;

  default:
    DPRINTK(1, "Unsupported depth %ld\n", p->depth);
    break;
  }

  base = p->base;

  var->xoffset = base % var->xres;
  var->yoffset = base / var->xres;

  var->height = var->width = -1;

  var->pixclock = KHZ2PICOS(p->pixclock);

  DPRINTK(3, "Clock done\n");

  if ((p->video & PM3VideoControl_HSYNC_MASK) == PM3VideoControl_HSYNC_ACTIVE_HIGH)
    var->sync |= FB_SYNC_HOR_HIGH_ACT;
  if ((p->video & PM3VideoControl_VSYNC_MASK) == PM3VideoControl_VSYNC_ACTIVE_HIGH)
    var->sync |= FB_SYNC_VERT_HIGH_ACT;
  if (p->video & PM3VideoControl_LINE_DOUBLE_ON)
    var->vmode = FB_VMODE_DOUBLE;

  DPRINTK(3, "all done\n");

  return(0);
}

static void pm3fb_get_par(void *par, struct fb_info_gen *info)
{
  struct pm3fb_info *l_fb_info = (struct pm3fb_info*) info;

  DTRACE;

  if (!current_par_valid[l_fb_info->board_num]) {
	if (l_fb_info->use_current)
	  pm3fb_read_mode(l_fb_info, l_fb_info->current_par);
	else
	  memcpy(l_fb_info->current_par, &(mode_base[0].user_mode), sizeof(struct pm3fb_par));
	current_par_valid[l_fb_info->board_num] = 1;
  }
  *((struct pm3fb_par* )par) = *(l_fb_info->current_par);
}

static void pm3fb_set_par(const void *par, struct fb_info_gen *info)
{
  struct pm3fb_info *l_fb_info = (struct pm3fb_info*) info;

  DTRACE;
  
  *(l_fb_info->current_par)  =  *((struct pm3fb_par*)par);
  current_par_valid[l_fb_info->board_num] = 1;

  pm3fb_write_mode(l_fb_info);
}

static void pm3fb_set_color(struct pm3fb_info* l_fb_info, unsigned char regno,
			    unsigned char r, unsigned char g, unsigned char b)
{
  DTRACE;

  DPRINTK(3, "Writing color #%d, rgb %3d:%3d:%3d\n", (int)regno, (int)r, (int)g, (int)b);

  PM3_SLOW_WRITE_REG(PM3RD_PaletteWriteAddress, regno);
  PM3_SLOW_WRITE_REG(PM3RD_PaletteData, r);
  PM3_SLOW_WRITE_REG(PM3RD_PaletteData, g);
  PM3_SLOW_WRITE_REG(PM3RD_PaletteData, b);
}

static int pm3fb_getcolreg(unsigned regno, unsigned *red, unsigned *green,
			   unsigned *blue, unsigned *transp,
			   struct fb_info *info)
{
  struct pm3fb_info* l_fb_info = (struct pm3fb_info* )info;
  
  DTRACE;

  if (regno<256) {
    *red = l_fb_info->palette[regno].red<<8|l_fb_info->palette[regno].red;
    *green = l_fb_info->palette[regno].green<<8|l_fb_info->palette[regno].green;
    *blue = l_fb_info->palette[regno].blue<<8|l_fb_info->palette[regno].blue;
    *transp = l_fb_info->palette[regno].transp<<8|l_fb_info->palette[regno].transp;
  }
  return(regno > 255);
}

static int pm3fb_setcolreg(unsigned regno, unsigned red, unsigned green,
			   unsigned blue, unsigned transp,
			   struct fb_info *info)
{
  struct pm3fb_info* l_fb_info = (struct pm3fb_info* )info;
  
  DTRACE;
  
  if (regno<16) {
    switch (l_fb_info->current_par->depth) {
#ifdef FBCON_HAS_CFB8
    case 8:
      break;
#endif
#ifdef FBCON_HAS_CFB16
    case 16:
      l_fb_info->cmap.cmap16[regno]=
	((u32 )red & 0xf800) |
	(((u32 )green & 0xfc00)>>5) |
	(((u32 )blue & 0xf800)>>11);
      break;
#endif
#ifdef FBCON_HAS_CFB32
    case 32:
      l_fb_info->cmap.cmap32[regno]=
	(((u32 )transp & 0xff00) << 16) |
	(((u32 )red & 0xff00) << 8) |
	(((u32 )green & 0xff00)) |
	(((u32 )blue & 0xff00) >> 8);
      break;
#endif
    default:
      DPRINTK(1, "bad depth %u\n", l_fb_info->current_par->depth);
      break;
    }
  }
  if (regno<256) {
    l_fb_info->palette[regno].red=red >> 8;
    l_fb_info->palette[regno].green=green >> 8;
    l_fb_info->palette[regno].blue=blue >> 8;
    l_fb_info->palette[regno].transp=transp >> 8;
    if (l_fb_info->current_par->depth==8)
      pm3fb_set_color(l_fb_info, regno, red>>8, green>>8, blue>>8);
  }
  return(regno > 255);
}

static int pm3fb_blank(int blank_mode, struct fb_info_gen *info)
{
  /* TO BE FINISHED */
#if 0
  struct pm3fb_info *l_fb_info = (struct pm3fb_info*) info;
  u32 video;

  DTRACE;
  
  if (!current_par_valid[l_fb_info->board_num])
    return(1);

  video = l_fb_info->current_par->video;

  if (blank_mode>0)
    {
      switch (blank_mode-1) {

      case VESA_NO_BLANKING: /* FIXME */
	video = video & ~(PM3VideoControl_ENABLE);
	break;

      case VESA_HSYNC_SUSPEND:
	video = video & ~(PM3VideoControl_HSYNC_MASK|
			  PM3VideoControl_BLANK_ACTIVE_LOW);
	break;
      case VESA_VSYNC_SUSPEND:
	video = video & ~(PM3VideoControl_VSYNC_MASK|
			  PM3VideoControl_BLANK_ACTIVE_LOW);
	break;
      case VESA_POWERDOWN:
	video = video & ~(PM3VideoControl_HSYNC_MASK|
			  PM3VideoControl_VSYNC_MASK|
			  PM3VideoControl_BLANK_ACTIVE_LOW);
	break;
      default:
	DPRINTK(1, "Unsupported blanking %d\n", blank_mode);
	return(1);
	break;
      }
    }

  PM3_SLOW_WRITE_REG(PM3VideoControl, video);

  return(0);
#endif /* 0 */
  return(1);
}

static void pm3fb_set_disp(const void *par, struct display *disp,
			   struct fb_info_gen *info)
{
  struct pm3fb_info *l_fb_info = (struct pm3fb_info *)info;
  struct pm3fb_par *p = (struct pm3fb_par *)par;
  u32 flags;

  DTRACE;
  
  save_flags(flags); cli();
  disp->screen_base=l_fb_info->v_fb;
  switch (p->depth)
    {
#ifdef FBCON_HAS_CFB8
    case 8:
      disp->dispsw=&fbcon_cfb8;
      break;
#endif
#ifdef FBCON_HAS_CFB16
    case 16:
      disp->dispsw=&fbcon_cfb16;
      disp->dispsw_data=l_fb_info->cmap.cmap16;
      break;
#endif
#ifdef FBCON_HAS_CFB32
    case 32:
      disp->dispsw=&fbcon_cfb32;
      disp->dispsw_data=l_fb_info->cmap.cmap32;
      break;
#endif
    default:
      disp->dispsw=&fbcon_dummy;
      DPRINTK(1, "Invalid depth, using fbcon_dummy\n");
      break;
    }
  restore_flags(flags);
}

/* */
static void pm3fb_detect(void)
{
  struct pci_dev* dev_array[PM3_MAX_BOARD];
  struct pci_dev* dev = NULL;
  struct pm3fb_info *l_fb_info = &(fb_info[0]);
  unsigned long i, j;
  
  DTRACE;

   for (i = 0 ; i < PM3_MAX_BOARD ; i++)
     {
       dev_array[i] = NULL;
       fb_info[i].dev = NULL;
     }

  dev = pci_find_device(PCI_VENDOR_ID_3DLABS, PCI_DEVICE_ID_3DLABS_PERMEDIA3, dev);

  for (i = 0 ; ((i < PM3_MAX_BOARD) && dev) ; i++)
    {
      dev_array[i] = dev;
      dev = pci_find_device(PCI_VENDOR_ID_3DLABS, PCI_DEVICE_ID_3DLABS_PERMEDIA3, dev);
    }
  
  if (dev) /* more than PM3_MAX_BOARD */
    {
      printk("pm3fb: Warning: more than %d boards found\n", PM3_MAX_BOARD);
    }

  if (!dev_array[0]) /* not a single board, abort */
    {
      return;
    }

  /* allocate user-defined boards */
  for (i = 0 ; i < PM3_MAX_BOARD ; i++)
    {
      if ((bus[i] >= 0) && (slot[i] >= 0) && (func[i] >= 0))
	{
	  for (j = 0; j < PM3_MAX_BOARD ; j++)
	    {
	      if ((dev_array[j] != NULL) &&
		  (dev_array[j]->bus->number == bus[i]) &&
		  (PCI_SLOT(dev_array[j]->devfn) == slot[i]) &&
		  (PCI_FUNC(dev_array[j]->devfn) == func[i]))
		{
		  fb_info[i].dev = dev_array[j];
		  dev_array[j] = NULL;
		}
	    }
	}
    }
  /* allocate remaining boards */
  for (i = 0 ; i < PM3_MAX_BOARD ; i++)
    {
      if (fb_info[i].dev == NULL)
	{
	  for (j = 0; j < PM3_MAX_BOARD ; j++)
	    {
	      if (dev_array[j] != NULL)
		{
		  fb_info[i].dev = dev_array[j];
		  dev_array[j] = NULL;
		}
	    }
	}
    }

  /* at that point, all PCI Permedia3 are detected and allocated */
  /* now, initialize... or not */
  for (i = 0 ; i < PM3_MAX_BOARD ; i++)
    {
      l_fb_info = &(fb_info[i]);
      if ((l_fb_info->dev) && (!disable[i]))/* PCI device was found and not disabled by user */
	{
#ifdef SUPPORT_FB_OF
	  struct device_node *dp =
	    find_pci_device_OFnode(l_fb_info->dev->bus->number, l_fb_info->dev->devfn);
	  
	  if ((dp) && (!strncmp(dp->name, "formacGA12",10)))
	    {
	      /* do nothing, init of board is done in pm3fb_of_init */
	    }
	  else
	    {
#endif
	  DPRINTK(2, "found @%lx Vendor %lx Device %lx ; base @ : %lx - %lx - %lx - %lx - %lx - %lx, irq %ld\n",
		  (unsigned long)l_fb_info->dev,
		  (unsigned long)l_fb_info->dev->vendor,
		  (unsigned long)l_fb_info->dev->device,
		  (unsigned long)pci_resource_start(l_fb_info->dev,0),
		  (unsigned long)pci_resource_start(l_fb_info->dev,1),
		  (unsigned long)pci_resource_start(l_fb_info->dev,2),
		  (unsigned long)pci_resource_start(l_fb_info->dev,3),
		  (unsigned long)pci_resource_start(l_fb_info->dev,4),
		  (unsigned long)pci_resource_start(l_fb_info->dev,5),
		  (unsigned long)l_fb_info->dev->irq);
	  
	  l_fb_info->pIOBase = (unsigned char*)pci_resource_start(l_fb_info->dev,0);
#ifdef __BIG_ENDIAN
	  l_fb_info->pIOBase += PM3_REGS_SIZE;
#endif
	  l_fb_info->vIOBase = (unsigned char*)-1;
	  l_fb_info->p_fb = (unsigned char*)pci_resource_start(l_fb_info->dev,1);
	  l_fb_info->v_fb = (unsigned char*)-1;

	  l_fb_info->fb_size = pm3fb_size_memory(l_fb_info);

#ifdef KERNEL_2_4 /* full resource management, new in linux-2.4.x */
	  if (!request_mem_region(l_fb_info->p_fb, l_fb_info->fb_size, "pm3fb"))
	    {
	      printk("pm3fb: Error: couldn't request framebuffer memory, board #%d\n", l_fb_info->board_num);
	      continue;
	    }
	  if (!request_mem_region(l_fb_info->pIOBase, PM3_REGS_SIZE, "pm3fb"))
	    {
	      printk("pm3fb: Error: couldn't request IObase memory, board #%d\n", l_fb_info->board_num);
	      continue;
	    }
#endif /* KERNEL_2_4 */

	  (void)pci_enable_device(l_fb_info->dev); /* #define'd to nothing in linux/kcomp.h in 2.2.18 */

	  pm3fb_common_init(l_fb_info);

#ifdef SUPPORT_FB_OF
	    }
#endif /* SUPPORT_FB_OF */
	}
    }
}

/* ****************************************** */
/* ***** standard FB API init functions ***** */
/* ****************************************** */

#ifdef KERNEL_2_4
int __init pm3fb_setup(char *options)
#endif
#ifdef KERNEL_2_2
__initfunc(void pm3fb_setup(char *options, int *ints))
#endif
{
  long opsi = strlen(options);

  DTRACE;

  memcpy(g_options, options, ((opsi+1) > PM3_OPTIONS_SIZE) ? PM3_OPTIONS_SIZE : (opsi + 1));
  g_options[PM3_OPTIONS_SIZE - 1] = 0;

#ifdef KERNEL_2_4
  return(0);
#endif
}

#ifdef KERNEL_2_4
int __init pm3fb_init(void)
#endif
#ifdef KERNEL_2_2
__initfunc(void pm3fb_init(void))
#endif
{
  DTRACE;

  pm3fb_real_setup(g_options);

  pm3fb_detect();

  if (!fb_info[0].dev) /* not even one board ??? */
    {
      DPRINTK(1, "No PCI Permedia3 board detected\n");
    }
  
#ifdef KERNEL_2_4
  return(0);
#endif
}

#ifdef SUPPORT_FB_OF /* linux-2.2.x only */
__initfunc(void pm3fb_of_init(struct device_node *dp))
{
  struct pm3fb_info *l_fb_info = NULL; /* &(fb_info[detected_board]); */
  unsigned long i;
  long bn = -1;
  struct device_node *dn;

  DTRACE;

  DPRINTK(2, "OpenFirmware board : %s\n", dp->full_name);

  for (i = 0; i < dp->n_addrs; i++)
    {
      DPRINTK(2, "MemRange : 0x%08x - 0x%x\n", dp->addrs[i].address, dp->addrs[i].size);
    }

  for (i = 0 ; i < PM3_MAX_BOARD ; i++) /* find which PCI board is the OF device */
    {
      if (fb_info[i].dev)
	{
	  dn = find_pci_device_OFnode(fb_info[i].dev->bus->number, fb_info[i].dev->devfn);
	  if (dn == dp)
	    {
	      if (bn == -1)
		bn = i;
	      else
		{
		  DPRINTK(1, "Error: Multiple PCI device for a single OpenFirmware node\n");
		}
	    }
	}
    }

  if (bn == -1)
    {
      DPRINTK(1, "Warning: non-PCI Permedia3 found\n");
      i = 0;
      while (fb_info[i].dev && (i < PM3_MAX_BOARD))
	i++;
      if (i < PM3_MAX_BOARD)
	bn = i;
      else
	{
	  printk("pm3fb: Error: Couldn't find room for OpenFirmware device");
	  return;
	}
    }
  
  l_fb_info = &(fb_info[bn]);

  l_fb_info->dn = dp;

  l_fb_info->pIOBase = (unsigned char*)dp->addrs[3].address;
#ifdef __BIG_ENDIAN
  l_fb_info->pIOBase += PM3_REGS_SIZE;
#endif
  l_fb_info->vIOBase = (unsigned char*)-1;
  l_fb_info->p_fb = (unsigned char*)dp->addrs[1].address;
  l_fb_info->v_fb = (unsigned char*)-1;
  
  l_fb_info->fb_size = pm3fb_size_memory(l_fb_info); /* (unsigned long)dp->addrs[1].size; */ /* OF is a liar ! it claims 256 Mb */

  DPRINTK(2, "OpenFirmware board (#%ld) : IOBase 0x%08lx, p_fb 0x%08lx, fb_size %d KB\n",
	  bn,
	  (unsigned long)l_fb_info->pIOBase, (unsigned long)l_fb_info->p_fb, l_fb_info->fb_size >> 10);

  l_fb_info->use_current = 1; /* will use current mode by default */

  pm3fb_common_init(l_fb_info);
}
#endif /* SUPPORT_FB_OF */

/* ************************* */
/* **** Module support ***** */
/* ************************* */

/* never tested yet ! */

#ifdef MODULE
int init_module(void)
{
  DTRACE;
 
  pm3fb_init();

  return(0);
}

void cleanup_module(void)
{
  DTRACE;
  {
    unsigned long i;
    pm3fb_info *l_fb_info;
    for (i = 0; i < PM3_MAX_BOARD; i++)
      {
	l_fb_info = &(fb_info[i]);
	if (l_fb_info->dev != NULL)
	  {
	    if (l_fb_info->vIOBase != (unsigned char*)-1)
	      {
		pm3fb_unmapIO(l_fb_info);
		release_mem_region(l_fb_info->p_fb, l_fb_info->fb_size);
		release_mem_region(l_fb_info->pIOBase, PM3_REGS_SIZE);
	      }
	    unregister_framebuffer(&l_fb_info->gen.info);
	  }
      }
  }
  return;
}
#endif /* MODULE */
